<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://designingmedia.com/
 * @since      1.0.0
 *
 * @package    Akd_Framework
 * @subpackage Akd_Framework/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Akd_Framework
 * @subpackage Akd_Framework/admin
 * @author     Ammar Nasir <info@domain.com>
 */
class Akd_Framework_Admin {

	use Akd_Theme_Activation;
	use Akd_Required_plugin_Activation;

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * The activated theme.
	 *
	 * @var string
	 */
	private $activated_theme;
	private $json_url;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since 1.0.0
	 * @param string $plugin_name The name of this plugin.
	 * @param string $version     The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name     = $plugin_name;
		$this->version         = $version;
		$this->activated_theme = wp_get_theme();
		$this->json_url = get_option( '_akd_download_url' ) . 'Hostiko/' . 'akd-importer-data-listing.json';
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Akd_Framework_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Akd_Framework_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/akd-framework-admin.css', array(), $this->version, 'all' );
		wp_enqueue_style( 'sweetalert', 'https://cdnjs.cloudflare.com/ajax/libs/sweetalert2/11.11.0/sweetalert2.min.css', array(), $this->version, 'all' );
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Akd_Framework_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Akd_Framework_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/akd-framework-admin.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( 'merlin', plugins_url() . '/akd-demo-importer/includes/merlin/assets/js/merlin.js', array( 'jquery' ), $this->version );
		wp_enqueue_script( 'sweetalert', 'https://cdnjs.cloudflare.com/ajax/libs/sweetalert2/11.11.0/sweetalert2.min.js', array( 'jquery' ), $this->version, false );

		if (is_admin() && isset($_GET['page']) && $_GET['page'] === 'akd-framework') {
			/**
			 * Localize the script with URLs
			 */
			$urls = array(
				'siteUrl' => site_url(),
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
			);

			$elementor = false;
			/**
			 * Check if Elementor plugin is active
			 */
			if ( in_array( 'elementor/elementor.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ), true ) ) {
				$elementor = true;
			}

			$additional_data = array();
			if ( get_option( '_pruchase_code' ) && get_option( '_encryption_key' ) ) {
				$additional_data = array(
					'purchaseCode' => $this->akd_decrypt_data( get_option( '_pruchase_code' ), get_option( '_encryption_key' ) ),
				);
			}

			/**
			 * Check if TGMPA is loaded (optional)
			 */
			if ( class_exists( 'TGM_Plugin_Activation' ) ) {
				/**
				 * Check if TGMPA is initialized
				 */
				if ( isset( $GLOBALS['tgmpa'] ) ) {
					/**
					 *  Get TGMPA instance
					 */
					$tgmpa_instance   = $GLOBALS['tgmpa'];
					$required_plugins = '';
					if ( true == $this->akd_check_required_plugins() ) {
						$required_plugins = 'installed';
					}
				}
			}

			$child_theme_directory = get_theme_root() . '/hostiko-child';

			$theme_info = array(
				'themeName'           => $this->activated_theme->get( 'Name' ),
				'childTheme'          => $this->activated_theme->get( 'Template' ),
				'elementor'           => $elementor,
				'requiredPlugins'     => $required_plugins,
				'childThemeFound'     => is_dir( $child_theme_directory ),
				'demoDataImported'    => get_option( '_akd_imported_layout' ),
				'licenseActivated'    => get_option( '_purchase_code_validate' ),
				'installedChildTheme' => is_child_theme(),
			);

			global $wpdb;

			/**
			 * Create an array to store all the information
			 */
			$server_info = array(
				'wp_version'             => get_bloginfo( 'version' ),
				'wp_dir_writeable'       => is_writable( wp_upload_dir()['basedir'] ),
				'php_v'                  => phpversion(),
				'php_memory_limit'       => ini_get( 'memory_limit' ),
				'php_post_max_size'      => ini_get( 'post_max_size' ),
				'php_max_upload_size'    => ini_get( 'upload_max_filesize' ),
				'php_max_input_vars'     => ini_get( 'max_input_vars' ),
				'php_max_execution_time' => ini_get( 'max_execution_time' ),
				'wp_memory_limit'        => WP_MEMORY_LIMIT,
				'mysql_v'                => $wpdb->db_version(),
				'file_get_content'       => $this->is_file_get_contents_active(),
				'curl_option'       	 => $this->is_curl_enabled(),
			);

			/**
			 * Combine localization data into a single array
			 */
			$localization_data = array(
				'urls_to_frontend'    => $urls,
				'additional_data'     => $additional_data,
				'theme_info'          => $theme_info,
				'server_info'         => $server_info,
				'plugin_path'		  => plugin_dir_url( __FILE__ ),
				'theme_demo'          => get_option( '_akd_imported_layout' ) ? get_option( '_akd_imported_layout' ) : '',
				'valid_purchase_code' => get_option( '_purchase_code_validate' ) ? get_option( '_purchase_code_validate' ) : '',
				'whmcs_url'           => get_option( 'whmcs_url' ),
				'whmcs_identifier'    => get_option( 'whmcs_identifier' ),
				'whmcs_secret'        => get_option( 'whmcs_secret' ),
			);

			/**
			 * Localize the script with combined data
			 */
			wp_localize_script( $this->plugin_name, 'akd_localization_data', $localization_data );

			wp_localize_script(
				'merlin',
				'merlin_params',
				array(
					'tgm_plugin_nonce' => array(
						'update'  => wp_create_nonce( 'tgmpa-update' ),
						'install' => wp_create_nonce( 'tgmpa-install' ),
					),
					'ajaxurl'  => admin_url( 'admin-ajax.php' ),
					'wpnonce'  => wp_create_nonce( 'merlin_nonce' ),
					'siteUrl'  => site_url(),
					'adminUrl' => admin_url(),
				)
			);
		}
	}

	public function is_curl_enabled() {
		if (function_exists('curl_version')) {
			$curl = curl_init();
			curl_setopt_array($curl, array(
				CURLOPT_URL => $this->json_url,
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_ENCODING => '',
				CURLOPT_MAXREDIRS => 10,
				CURLOPT_TIMEOUT => 3,
				CURLOPT_FOLLOWLOCATION => true,
				CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
				CURLOPT_CUSTOMREQUEST => 'GET',
			));

			$json_content = curl_exec($curl);

			curl_close($curl);
				if ( FALSE === $json_content ) {
					return 'Disable';
				}else{
					return 'Enable';
				}
			}else{
				return 'Disable';
			}
	}

	public function is_file_get_contents_active() {

		$allow_url_fopen = ini_get('allow_url_fopen');
		if ($allow_url_fopen) {
			$data = file_get_contents($this->json_url);
			if( $data ) {
				return 'Enable';
			}
			else{
				return 'Disable';
			}
		} else {
			return 'Disable';
		}
	}

	public function pre( $data ) {
		echo '<pre>'; print_r( $data );
	}

	/**
	 * Method to check if required plugins are installed and activated
	 */
	public function akd_check_required_plugins() {
		$tgmpa = TGM_Plugin_Activation::get_instance();

		$required_plugins = array();
		foreach ( $tgmpa->plugins as $slug => $plugin ) {
			if ( ! empty( $plugin['required'] ) && true == $plugin['required'] ) {
				$required_plugins[] = $slug;
			}
		}

		foreach ( $required_plugins as $slug ) {
			$plugin_data = $tgmpa->plugins[ $slug ];
			$plugin_path = $plugin_data['file_path'];

			/**
			 * Check if the plugin files exist
			 */
			$is_installed = file_exists( WP_PLUGIN_DIR . '/' . $plugin_path );

			/**
			 * Check if the plugin is active
			 */
			$is_active = is_plugin_active( $plugin_path );

			if ( ! $is_installed || ! $is_active ) {
				/**
				 * If a required plugin is not installed or not activated, return false
				 */
				return false;
			}
		}

		/**
		 * If all required plugins are installed and activated, return true
		 */
		return true;
	}


	/**
	 * Method to trigger admin_init hook methods
	 */
	public function akd_admin_init_callback() {
		/**
		 * Method to remove GDPR cookie compliance warnings from the merlin page
		 */
		$this->akd_remove_cookie_compliance_warnings();

		/**
		 * Method to get and save theme brandings like logo and fvicons
		 */
		$this->akd_get_theme_branding_settings();


		/**
		 * Method to get and save elementor global settings within wp_options
		 */
		$this->akd_get_elementor_global_settings();

		/**
		 * Method to process manual import
		 */
		$this->process_manual_import();

		// apply validation for single layout present
		if ( get_option( '_akd_demo_import' ) ) {
			$this->akd_delete_layout_folders_after_import();
		}

		if ( isset( $_GET['page'] ) && isset( $_GET['path'] ) && isset( $_SESSION['plugin_redirect'] ) ) {
			if ( 'wc-admin' === $_GET['page'] && '/setup-wizard' === $_GET['path'] && true == $_SESSION['plugin_redirect'] ) {
				$_SESSION['plugin_redirect'] = false;
				wp_redirect( admin_url( 'admin.php?page=akd-framework#tab=theme-dashboard' ) ); exit;
			}
		}

		if ( isset( $_GET['page'] ) && isset( $_SESSION['plugin_redirect'] ) ) {
			if ( 'revslider' === $_GET['page'] && true == $_SESSION['plugin_redirect'] ) {
				$_SESSION['plugin_redirect'] = false;
				wp_redirect( admin_url( 'admin.php?page=akd-framework#tab=theme-dashboard' ) ); exit;
			}
		}
	}

	/**
	 * Method to check and delete layout-* folders
	 */
	public function akd_delete_layout_folders_after_import() {
		/**
		 * Get the uploads directory path
		 */
		$uploads_dir = wp_upload_dir()['path'];

		/**
		 * Check if the directory exists
		 */
		if ( is_dir( $uploads_dir ) ) {
			/**
			 * Scan the directory for subfolders matching the pattern 'layout-*'
			 */
			$subfolders = array_filter( glob( $uploads_dir . '/layout-*'), 'is_dir' );

			/**
			 * If there is more than one 'layout-*' folder, proceed to delete older ones
			 */
			if ( count( $subfolders ) > 1 ) {
				// Sort the subfolders by modification time, newest first
				usort( $subfolders, function( $a, $b ) {
					return filemtime( $b ) - filemtime( $a );
				});

				// Keep the latest folder (newest) and delete the others
				foreach ( array_slice( $subfolders, 1 ) as $folder ) {
					$this->delete_dir( $folder );
				}
			}
		}

	}


	/**
	 * Method to remove files from the directory.
	 *
	 * @param string $dir The directory from which to remove files.
	 */
	private function delete_dir( $dir ) {
		if ( ! file_exists( $dir ) ) {
			return true;
		}

		if ( ! is_dir( $dir ) || is_link( $dir ) ) {
			return unlink( $dir );
		}

		foreach ( scandir( $dir ) as $item ) {
			if ( '.' == $item || '..' == $item) {
				continue;
			}

			if ( ! $this->delete_dir( $dir . DIRECTORY_SEPARATOR . $item ) ) {
				return false;
			}
		}

		return rmdir( $dir );
	}

	/**
	 * Method to process manual import
	 */
	private function process_manual_import() {
		$demo_directory = plugin_dir_path( dirname( __FILE__ ) ) . 'admin/demo/';
		$zip_files = glob( $demo_directory . '*.zip' );

		if ( ! empty( $zip_files ) ) {
			foreach ( $zip_files as $zip_file ) {

				/**
				 * Extract the folder name from the zip file name
				 */
				$folder_name = pathinfo( $zip_file, PATHINFO_FILENAME );

				if ( strpos( $folder_name, 'manual-' ) === 0 ) {
					/**
					 * Remove the "manual-" prefix
					 */
					$folder_name       = substr( $folder_name, 7 );
					$new_zip_file_name = $folder_name . '.zip';
					$new_zip_file_path = $demo_directory . $new_zip_file_name;

					/**
					 * Rename the zip file
					 */
					rename( $zip_file, $new_zip_file_path );
				}

				/**
				 * Check if the folder exists, and remove it
				 */
				$existing_folder = $demo_directory . $folder_name;
				if ( is_dir( $existing_folder ) ) {
					/**
					 * Remove existing folder and its contents
					 */
					$this->delete_dir( $existing_folder );
				}

				$zip_file = $new_zip_file_path;

				/**
				 * Unzip the file
				 */
				$zip = new ZipArchive;
				if ( $zip->open( $zip_file ) === TRUE ) {
					$zip->extractTo( $demo_directory );
					$zip->close();

					/**
					 * Remove the zip file
					 */
					unlink( $zip_file );

					/**
					 * Get the path to the extracted uploads folder
					 */
					$extracted_uploads_folder = $demo_directory . $folder_name . '/upload/';

					/**
					 * Check if the uploads folder exists
					 */
					if ( is_dir( $extracted_uploads_folder ) ) {
						/**
						 * Get list of all directories within the uploads folder
						 */
						$directories = glob( $extracted_uploads_folder . '*', GLOB_ONLYDIR );

						/**
						 * Move each directory to the WordPress uploads directory
						 */
						foreach ( $directories as $directory ) {
							$destination = WP_CONTENT_DIR . '/uploads/' . basename( $directory );

							if ( file_exists( $destination ) ) {
								/**
								 * Remove existing destination folder
								 */
								$this->delete_dir( $destination );
							}

							/**
							 * Move the directory
							 */
							rename( $directory, $destination );

							/**
							 * Replace URLs in XML files
							 */
							$xml_file = $demo_directory . '' . $folder_name . '/hostiko-' . preg_replace( '/-/', '', $folder_name ) . '-content.xml';
							if ( file_exists( $xml_file ) ) {
								$xml_content      = file_get_contents( $xml_file );
								$current_site_url = site_url() . '/wp-content/uploads/';
								$xml_content      = str_replace( 'https://hostiko.com/' . preg_replace( '/-/', '', $folder_name ) . '/wp-content/uploads/', $current_site_url, $xml_content );

								file_put_contents( $xml_file, $xml_content );
							}
						}

						/**
						 * Move the extracted folder to the current uploads directory
						 */
						$this->move_extracted_folder( $demo_directory, $folder_name );
					}
				}
			}
		}
	}

	/**
	 * Move the extracted folder to the current uploads directory.
	 *
	 * @param string $demo_directory The path to the demo directory.
	 * @param string $folder_name    The name of the folder to move.
	 */
	private function move_extracted_folder( $demo_directory, $folder_name ) {
		$source      = $demo_directory . $folder_name;
		$upload_dir  = wp_upload_dir();
		$destination = $upload_dir['path'] . '/' . $folder_name;

		if ( is_dir( $source ) ) {
			if ( file_exists( $destination ) ) {
				/**
				 * Remove existing destination folder
				 */
				$this->delete_dir( $destination );
			}
			rename( $source, $destination );
		}
	}

	/**
	 * Method to remove GDPR cookie compliance warnings from the merlin page
	 */
	private function akd_remove_cookie_compliance_warnings() {
		/**
		 * Cookie complaince plugin warning on merlin page
		 */
		if ( is_plugin_active( 'gdpr-cookie-compliance/moove-gdpr.php' ) ) {
			if ( isset( $_GET['page'] ) && 'merlin' === $_GET['page'] ) {
				$this->remove_class_action( 'gdpr_check_review_banner_condition', 'Moove_GDPR_Review', 'gdpr_check_review_banner_condition_func' );
			}
		}
	}

	/**
	 * Method to import demo content files
	 */
	public function akd_import_theme_demo_files() {

		$demo_dir = plugin_dir_path( dirname( __FILE__ ) ) . 'admin/demo/';
		if ( is_dir( $demo_dir ) ) {
			$files = array_diff( scandir( $demo_dir ), array('.', '..') );
			/**
			 * Hostiko template files path
			 */
			if ( count( $files ) === 1 && in_array( 'index.php', $files ) ) {
				$uploads_dir = wp_upload_dir()['path'];

				/**
				 * Check if the directory exists
				 */
				if ( is_dir( $uploads_dir ) ) {
					/**
					 * Scan the directory for subfolders matching the pattern 'layout-*'
					 */
					$subfolders = array_filter( glob( $uploads_dir . '/layout-*' ), 'is_dir' );

					/**
					 * Sort the subfolders by modification time, most recent first
					 */
					usort( $subfolders, function( $a, $b ) {
						return filemtime( $b ) - filemtime( $a );
					});

					/**
					 * Get the most recently modified layout folder
					 */
					$latest_layout = reset( $subfolders );

					/**
					 * Initialize an empty array to hold the layout data
					 */
					$layouts = array();

					/**
					 * Loop through each found subfolder
					 */

					/**
					 * Extract the layout number from the folder name
					 */
					if ( preg_match( '/layout-(\d+)/', basename( $latest_layout ), $matches ) ) {
						$layout_number = $matches[1];

						$slider_folder_path = wp_upload_dir()['path'] . '/layout-' . $layout_number . '/sliders';
						$zip_files          = glob( $slider_folder_path . '/*.zip' );

						if ( ! empty( $zip_files ) ) {
							/**
							 * Create the array entry for the current layout
							 */
							$layouts[] = array(
								'import_file_name'             => 'Hostiko Layout ' . $layout_number,
								'categories'                   => array(),
								'local_import_file'            => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/hostiko-layout' . $layout_number . '-content.xml',
								'local_import_widget_file'     => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/site-settings.json',
								'local_import_customizer_file' => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/hostiko-customizer.dat',
								'local_import_options_file'    => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/exported-options.json',
								'import_preview_image_url'     => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/screenshot.png',
								'local_import_rev_slider_file' => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/sliders',
								// 'local_import_rev_slider_file' => plugin_dir_path( __DIR__ ) . 'admin/demo/sliders/slaido_441.zip', 
							);
						} else {
							/**
							 * Create the array entry for the current layout
							 */
							$layouts[] = array(
								'import_file_name'             => 'Hostiko Layout ' . $layout_number,
								'categories'                   => array(),
								'local_import_file'            => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/hostiko-layout' . $layout_number . '-content.xml',
								'local_import_widget_file'     => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/site-settings.json',
								'local_import_customizer_file' => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/hostiko-customizer.dat',
								'local_import_options_file'    => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/exported-options.json',
								'import_preview_image_url'     => wp_upload_dir()['path'] . '/layout-' . $layout_number . '/screenshot.png',
							);
						}
					}
				} else {
					echo 'Uploads directory does not exist.';
				}

				return $layouts;
			} else {
				$activated_theme = wp_get_theme();
				$theme_name      = explode( ' ', $activated_theme->get( 'Name' ) )[0];
				return array(
					array(
						'import_file_name' => $theme_name,
						'categories' => array() ,
						'local_import_file'            => trailingslashit(plugin_dir_path( dirname( __FILE__ ) )).'admin/demo/insurerity-demo-content.xml',
						'local_import_widget_file'     => trailingslashit(plugin_dir_path( dirname( __FILE__ ) )).'admin/demo/site-settings.json',
						'local_import_customizer_file' => trailingslashit(plugin_dir_path( dirname( __FILE__ ) )).'admin/demo/insureritycustomizer.dat',        
						'import_preview_image_url'     => trailingslashit(plugin_dir_path( dirname( __FILE__ ) )).'admin/demo/screenshot.png',
					),
				);
			}
		}
	}

	/**
	 * Remove a specific action or filter from a class.
	 *
	 * This method removes the specified action or filter hook associated with a
	 * class method.
	 *
	 * @param string $action    The name of the action or filter to remove.
	 * @param string $classname The class that the action or filter is associated with.
	 * @param string $method    The method within the class that the action or filter is associated with.
	 */
	private function remove_class_action( $action, $classname, $method ) {
		global $wp_filter;
		if ( isset( $wp_filter[ $action ] ) ) {
			$len = strlen( $method );
			foreach ( $wp_filter[ $action ] as $pri => $actions ) {
				foreach ( $actions as $name => $def ) {
					if ( substr( $name, -$len ) == $method ) {
						if ( is_array( $def['function'] ) ) {
							if ( get_class( $def['function'][0] ) == $classname ) {
								if ( is_object( $wp_filter[$action] ) && isset( $wp_filter[ $action ]->callbacks ) ) {
									unset( $wp_filter[ $action ]->callbacks[ $pri ][ $name ] );
								} else {
									unset( $wp_filter[ $action ][ $pri ][ $name ] );
								}
							}
						}
					}
				}
			}
		}
	}

	/**
	 * Method to get theme logo and favicon  settings
	 */
	private function akd_get_theme_branding_settings() {
		if ( isset( $_POST['akd-framework-admin-page'] ) ) {
			if(isset($_POST['akd-framework-admin-page']['akd_update_logo'])){
				$theme = wp_get_theme();
				$theme_slug = $theme->get_stylesheet();
				$akd_theme_mods = get_option( 'theme_mods_' . $theme_slug );
				set_theme_mod( 'logo', $_POST['akd-framework-admin-page']['akd_update_logo'] );
			}

			if(isset($_POST['akd-framework-admin-page']['akd_update_favicon'])){
				$post_id = attachment_url_to_postid($_POST['akd-framework-admin-page']['akd_update_favicon']);
				// echo 'post_id: <pre>' . print_r($post_id, true) . '</pre>';die;
				update_option( 'site_icon', $post_id );
				
			}
		}

		// Update the top bar switch setting
		if ( isset( $_POST['akd-framework-admin-page'] ) ) {
			$theme = wp_get_theme();
			$theme_slug = $theme->get_stylesheet();
			$akd_theme_mods = get_option( 'theme_mods_' . $theme_slug );

			//Define the fields to check for update
			$fields = [
				'opt-switcher-topbar' => 'hostiko_topbar_switch',
				'opt-switcher-showbutton' => 'hostiko_show_header_btn',
				'login-button-txt' => 'hostiko_header_btn_text',
				'opt-switcher-topbar-bg' => 'hostiko_header_topbar_bg',
				
				'opt-switcher-livechat' => 'hostiko_show_live_chat_btn',
				'opt-livechat-bg' => 'hostiko_live_bg',
				'opt-livechat-color' => 'hostiko_live_color',
				'opt-livechat-border-color' => 'hostiko_live_border',
				'opt-livechat-border-radius' => 'hostiko_live_chat_br',
				'opt-livechat-padding' => 'hostiko_live_chat_padding',
				'opt-livechat-border-hover-color' => 'hostiko_live_border_hover',
				'opt-livechat-bg-hover-color' => 'hostiko_live_bg_hover',
				'opt-livechat-hover-color' => 'hostiko_live_color_hover',
			
				'opt-switcher-carticon' => 'hostiko_show_cart_icon_btn',
				'opt-carticon-bg' => 'hostiko_cart_icon_bg',
				'opt-carticon-color' => 'hostiko_cart_icon_color',
				'opt-carticon-border-color' => 'hostiko_cart_icon_border',
				'opt-carticon-border-radius' => 'hostiko_cart_icon_br',
				'opt-carticon-padding' => 'hostiko_cart_icon_padding',
				'opt-carticon-border-hover-color' => 'hostiko_cart_icon_border_hover',
				'opt-carticon-bg-hover-color' => 'hostiko_cart_icon_bg_hover',
				'opt-carticon-hover-color' => 'hostiko_cart_icon_color_hover',
				'opt-cartcounter-bg-color' => 'hostiko_cart_counter_bg',
				'opt-cartcounter-color' => 'hostiko_cart_counter_color',
				'opt-cartcounter-top' => 'hostiko_cart_counter_top',
				'opt-cartcounter-right' => 'hostiko_cart_counter_right',
			
				'opt-navbar-color' => 'hostiko_nav_color',
				'opt-navbar-hover-color' => 'hostiko_nav_hover_color',
				'opt-navbar-active-color' => 'hostiko_nav_active_color',
				'opt-navbar-dropdown-bg' => 'hostiko_nav_dropdown_bg',
				'opt-navbar-dropdown-hover-bg' => 'hostiko_nav_dropdown_bg_hover',
				'opt-navbar-dropdown-color' => 'hostiko_nav_dropdown_color',
				'opt-navbar-dropdown-hover-color' => 'hostiko_nav_dropdown_hover_color',
				'opt-navbar-dropdown-position' => 'hostiko_navbar_dropdown_position',
				'opt-navbar-dropdown-border-radius' => 'hostiko_navbar_dropdown_br',
				'opt-navbar-dropdown-border-width' => 'hostiko_dropdown_border_width',
				'opt-navbar-dropdown-border-color' => 'hostiko_nav_dropdown_border_color',
				'opt-navbar-dropdown-bg-color-mobile' => 'hostiko_nav_dropdown_bg_mobile',
				'opt-navbar-dropdown-color-mobile' => 'hostiko_nav_dropdown_color_mobile',
				'opt-navbar-dropdown-color-hover-mobile' => 'hostiko_nav_dropdown_color_hover_mobile',
				'opt-navbar-hamburger-color' => 'hostiko_nav_hamburger_color',
				'opt-navbar-dropdown-border-bottom-width' => 'hostiko_nav_border_bottom',
				
				'login-btn-link' => 'hostiko_header_btn_link',
				'opt-switcher-stikyheader' => 'hostiko_sticky_header_btn',
				'opt-switcher-stikyheader-bg' => 'hostiko_stickyheader_bg',
				'opt-select-header-elementor-template' => 'choose_elementor_header',
				'opt-select-footer-elementor-template' => 'choose_elementor_footer',
				'opt-select-footer-style' => 'choose_default_footer',
				'copyright-txt' => 'hostiko_copyright',
				'opt-switcher-subbanner' => 'hostiko_sub_banner_show',
				
				'opt-switcher-breadcrumb' => 'hostiko_breadcrumb_show',
				'opt-breadcrumb-title-color' => 'page_title_color',
				'opt-subtitle-color' => 'page_subtitle_color',
				'opt-breadcrumb-bg-color' => 'breadcrumb_bg_color',
				'opt-breadcrumb-color' => 'hostiko_breadcrumb_color',
				'opt-breadcrumb-current-page-color' => 'hostiko_breadcrumb_current_page_color',
				'opt-breadcrumb-hover-color' => 'hostiko_breadcrumb_hover_color',
				'opt-breadcrumb-active-color' => 'hostiko_breadcrumb_active_color',
				'opt-breadcrumb-border-radius' => 'hostiko_breadcrumb_br',
				'opt-breadcrumb-border-width' => 'hostiko_breadcrumb_border_width',
				'opt-breadcrumb-border-color' => 'hostiko_breadcrumb_border_color',
				'opt-breadcrumb-padding' => 'hostiko_breadcrumb_padding',
				'opt-subBannerImage' => 'sub-banner-img',
				'opt-subbanner-padding-desktop' => 'hostiko_sub_banner_padding_desk',
				'opt-subbanner-padding-laptop' => 'hostiko_sub_banner_padding_lap',
				'opt-subbanner-padding-tablet' => 'hostiko_sub_banner_padding_tab',
				'opt-subbanner-padding-mobile' => 'hostiko_sub_banner_padding_mobile',
				'opt-blog-page-title' => 'breadcrumb_blog_title',
				'opt-blog-page-subTitle' => 'breadcrumb_blog_title_details',
				'opt-404-title' => 'hostiko_error_404_text',
				'opt-404-not-found-title' => 'hostiko_error_title',
				'opt-404-desc' => 'hostiko_error_desc',
				'opt-404-link-text' => 'hostiko_error_link_text',
				'opt-switcher-preloader' => 'hostiko_preloader_switch',
				'opt-switcher-showinvestment' => 'hostiko_show_investment_offer_link',
				'investment-button-txt' => 'hostiko_header_link_text',
				'investment-btn-link' => 'hostiko_header_link_url',
				'opt-switcher-show-contact-info' => 'hostiko_show_contact_info',
				'email-txt' => 'hostiko_header_email',
				'phone-txt' => 'hostiko_header_phone',
				'opt-switcher-customizer-settings' => 'hostiko_enable_customizer_settings',
				'opt-switcher-socialprofiles' => 'hostiko_show_social_profiles',
				'social-fb-link' => 'hostiko_topbar_fb_url',
				'social-x-link' => 'hostiko_topbar_twitter_url',
				'social-linkedin-link' => 'hostiko_topbar_linkedin_url',
				'social-instagram-link' => 'hostiko_topbar_instagram_url',

				
			];


			$updated = false;

			foreach ($fields as $post_key => $theme_mod_key) {
				if (isset($_POST['akd-framework-admin-page'][$post_key])) {
					if(
						$post_key == 'login-btn-link' ||
						$post_key == 'investment-btn-link' ||
						$post_key == 'social-fb-link' ||
						$post_key == 'social-x-link' ||
						$post_key == 'social-linkedin-link' ||
						$post_key == 'social-instagram-link'
					)
					{
						$new_value = $_POST['akd-framework-admin-page'][$post_key]['url'];
					}else{
						$new_value = $_POST['akd-framework-admin-page'][$post_key];
					}
					
					if (!isset($akd_theme_mods[$theme_mod_key]) || $akd_theme_mods[$theme_mod_key] !== $new_value) {
						if($theme_mod_key == 'hostiko_preloader_switch'){
							$akd_theme_mods['show_preloader'] = $new_value;
						}
						$akd_theme_mods[$theme_mod_key] = $new_value;
						$updated = true;
					}
				}
			}

			// Update theme mods only if changes exist
			if ($updated) {
				// update_option('theme_mods_' . $theme_slug, $akd_theme_mods);
			}

			// Update the hostiko_topbar_switch setting in the array
			// $akd_theme_mods['hostiko_topbar_switch'] = isset($_POST['akd-framework-admin-page']['opt-switcher-topbar']) ? $_POST['akd-framework-admin-page']['opt-switcher-topbar'] : '';
			// $akd_theme_mods['hostiko_show_header_btn'] = isset($_POST['akd-framework-admin-page']['opt-switcher-showbutton']) ? $_POST['akd-framework-admin-page']['opt-switcher-showbutton'] : '';
			// $akd_theme_mods['hostiko_header_btn_text'] = isset($_POST['akd-framework-admin-page']['login-button-txt']) ? $_POST['akd-framework-admin-page']['login-button-txt'] : '';
			// $akd_theme_mods['hostiko_header_topbar_bg'] = isset($_POST['akd-framework-admin-page']['opt-switcher-topbar-bg']) ? $_POST['akd-framework-admin-page']['opt-switcher-topbar-bg'] : '';
			
			// $akd_theme_mods['hostiko_show_live_chat_btn'] = isset($_POST['akd-framework-admin-page']['opt-switcher-livechat']) ? $_POST['akd-framework-admin-page']['opt-switcher-livechat'] : '';
			// $akd_theme_mods['hostiko_live_bg'] = isset($_POST['akd-framework-admin-page']['opt-livechat-bg']) ? $_POST['akd-framework-admin-page']['opt-livechat-bg'] : '';
			// $akd_theme_mods['hostiko_live_color'] = isset($_POST['akd-framework-admin-page']['opt-livechat-color']) ? $_POST['akd-framework-admin-page']['opt-livechat-color'] : '';
			// $akd_theme_mods['hostiko_live_border'] = isset($_POST['akd-framework-admin-page']['opt-livechat-border-color']) ? $_POST['akd-framework-admin-page']['opt-livechat-border-color'] : '';
			// $akd_theme_mods['hostiko_live_chat_br'] = isset($_POST['akd-framework-admin-page']['opt-livechat-border-radius']) ? $_POST['akd-framework-admin-page']['opt-livechat-border-radius'] : '';
			// $akd_theme_mods['hostiko_live_chat_padding'] = isset($_POST['akd-framework-admin-page']['opt-livechat-padding']) ? $_POST['akd-framework-admin-page']['opt-livechat-padding'] : '';
			// $akd_theme_mods['hostiko_live_border_hover'] = isset($_POST['akd-framework-admin-page']['opt-livechat-border-hover-color']) ? $_POST['akd-framework-admin-page']['opt-livechat-border-hover-color'] : '';
			// $akd_theme_mods['hostiko_live_bg_hover'] = isset($_POST['akd-framework-admin-page']['opt-livechat-bg-hover-color']) ? $_POST['akd-framework-admin-page']['opt-livechat-bg-hover-color'] : '';
			// $akd_theme_mods['hostiko_live_color_hover'] = isset($_POST['akd-framework-admin-page']['opt-livechat-hover-color']) ? $_POST['akd-framework-admin-page']['opt-livechat-hover-color'] : '';
			
			// $akd_theme_mods['hostiko_show_cart_icon_btn'] = isset($_POST['akd-framework-admin-page']['opt-switcher-carticon']) ? $_POST['akd-framework-admin-page']['opt-switcher-carticon'] : '';
			// $akd_theme_mods['hostiko_cart_icon_bg'] = isset($_POST['akd-framework-admin-page']['opt-carticon-bg']) ? $_POST['akd-framework-admin-page']['opt-carticon-bg'] : '';
			// $akd_theme_mods['hostiko_cart_icon_color'] = isset($_POST['akd-framework-admin-page']['opt-carticon-color']) ? $_POST['akd-framework-admin-page']['opt-carticon-color'] : '';
			// $akd_theme_mods['hostiko_cart_icon_border'] = isset($_POST['akd-framework-admin-page']['opt-carticon-border-color']) ? $_POST['akd-framework-admin-page']['opt-carticon-border-color'] : '';
			// $akd_theme_mods['hostiko_cart_icon_br'] = isset($_POST['akd-framework-admin-page']['opt-carticon-border-radius']) ? $_POST['akd-framework-admin-page']['opt-carticon-border-radius'] : '';
			// $akd_theme_mods['hostiko_cart_icon_padding'] = isset($_POST['akd-framework-admin-page']['opt-carticon-padding']) ? $_POST['akd-framework-admin-page']['opt-carticon-padding'] : '';
			// $akd_theme_mods['hostiko_cart_icon_border_hover'] = isset($_POST['akd-framework-admin-page']['opt-carticon-border-hover-color']) ? $_POST['akd-framework-admin-page']['opt-carticon-border-hover-color'] : '';
			// $akd_theme_mods['hostiko_cart_icon_bg_hover'] = isset($_POST['akd-framework-admin-page']['opt-carticon-bg-hover-color']) ? $_POST['akd-framework-admin-page']['opt-carticon-bg-hover-color'] : '';
			// $akd_theme_mods['hostiko_cart_icon_color_hover'] = isset($_POST['akd-framework-admin-page']['opt-carticon-hover-color']) ? $_POST['akd-framework-admin-page']['opt-carticon-hover-color'] : '';
			// $akd_theme_mods['hostiko_cart_counter_bg'] = isset($_POST['akd-framework-admin-page']['opt-cartcounter-bg-color']) ? $_POST['akd-framework-admin-page']['opt-cartcounter-bg-color'] : '';
			// $akd_theme_mods['hostiko_cart_counter_color'] = isset($_POST['akd-framework-admin-page']['opt-cartcounter-color']) ? $_POST['akd-framework-admin-page']['opt-cartcounter-color'] : '';
			// $akd_theme_mods['hostiko_cart_counter_top'] = isset($_POST['akd-framework-admin-page']['opt-cartcounter-top']) ? $_POST['akd-framework-admin-page']['opt-cartcounter-top'] : '';
			// $akd_theme_mods['hostiko_cart_counter_right'] = isset($_POST['akd-framework-admin-page']['opt-cartcounter-right']) ? $_POST['akd-framework-admin-page']['opt-cartcounter-right'] : '';
			
			// $akd_theme_mods['hostiko_nav_color'] = isset($_POST['akd-framework-admin-page']['opt-navbar-color']) ? $_POST['akd-framework-admin-page']['opt-navbar-color'] : '';
			// $akd_theme_mods['hostiko_nav_hover_color'] = isset($_POST['akd-framework-admin-page']['opt-navbar-hover-color']) ? $_POST['akd-framework-admin-page']['opt-navbar-hover-color'] : '';
			// $akd_theme_mods['hostiko_nav_active_color'] = isset($_POST['akd-framework-admin-page']['opt-navbar-active-color']) ? $_POST['akd-framework-admin-page']['opt-navbar-active-color'] : '';
			// $akd_theme_mods['hostiko_nav_dropdown_bg'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-bg']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-bg'] : '';
			// $akd_theme_mods['hostiko_nav_dropdown_bg_hover'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-hover-bg']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-hover-bg'] : '';
			// $akd_theme_mods['hostiko_nav_dropdown_color'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-color']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-color'] : '';
			// $akd_theme_mods['hostiko_nav_dropdown_hover_color'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-hover-color']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-hover-color'] : '';
			// $akd_theme_mods['hostiko_navbar_dropdown_position'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-position']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-position'] : '';
			// $akd_theme_mods['hostiko_navbar_dropdown_br'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-border-radius']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-border-radius'] : '';
			// $akd_theme_mods['hostiko_dropdown_border_width'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-border-width']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-border-width'] : '';
			// $akd_theme_mods['hostiko_nav_dropdown_border_color'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-border-color']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-border-color'] : '';
			// $akd_theme_mods['hostiko_nav_dropdown_bg_mobile'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-bg-color-mobile']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-bg-color-mobile'] : '';
			// $akd_theme_mods['hostiko_nav_dropdown_color_mobile'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-color-mobile']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-color-mobile'] : '';
			// $akd_theme_mods['hostiko_nav_dropdown_color_hover_mobile'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-color-hover-mobile']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-color-hover-mobile'] : '';
			// $akd_theme_mods['hostiko_nav_hamburger_color'] = isset($_POST['akd-framework-admin-page']['opt-navbar-hamburger-color']) ? $_POST['akd-framework-admin-page']['opt-navbar-hamburger-color'] : '';
			// $akd_theme_mods['hostiko_nav_border_bottom'] = isset($_POST['akd-framework-admin-page']['opt-navbar-dropdown-border-bottom-width']) ? $_POST['akd-framework-admin-page']['opt-navbar-dropdown-border-bottom-width'] : '';
			// $akd_theme_mods['hostiko_nav_border_bottom_color'] = isset($_POST['akd-framework-admin-page']['opt-navbar-border-bottom-color']) ? $_POST['akd-framework-admin-page']['opt-navbar-border-bottom-color'] : '';
			
			
			// $akd_theme_mods['hostiko_header_btn_link'] = isset($_POST['akd-framework-admin-page']['login-btn-link']['url']) ? $_POST['akd-framework-admin-page']['login-btn-link']['url'] : '';
			// $akd_theme_mods['hostiko_sticky_header_btn'] = isset($_POST['akd-framework-admin-page']['opt-switcher-stikyheader']) ? $_POST['akd-framework-admin-page']['opt-switcher-stikyheader'] : '';
			// $akd_theme_mods['choose_elementor_header'] = isset($_POST['akd-framework-admin-page']['opt-select-header-elementor-template']) ? $_POST['akd-framework-admin-page']['opt-select-header-elementor-template'] : '';
			// $akd_theme_mods['choose_elementor_footer'] = isset($_POST['akd-framework-admin-page']['opt-select-footer-elementor-template']) ? $_POST['akd-framework-admin-page']['opt-select-footer-elementor-template'] : '';
			// $akd_theme_mods['choose_default_footer'] = isset($_POST['akd-framework-admin-page']['opt-select-footer-style']) ? $_POST['akd-framework-admin-page']['opt-select-footer-style'] : '';
			// $akd_theme_mods['hostiko_copyright'] = isset($_POST['akd-framework-admin-page']['copyright-txt']) ? $_POST['akd-framework-admin-page']['copyright-txt'] : '';
			// $akd_theme_mods['hostiko_sub_banner_show'] = isset($_POST['akd-framework-admin-page']['opt-switcher-subbanner']) ? $_POST['akd-framework-admin-page']['opt-switcher-subbanner'] : '';
			// $akd_theme_mods['hostiko_breadcrumb_show'] = isset($_POST['akd-framework-admin-page']['opt-switcher-breadcrumb']) ? $_POST['akd-framework-admin-page']['opt-switcher-breadcrumb'] : '';
			// $akd_theme_mods['page_title_color'] = isset($_POST['akd-framework-admin-page']['opt-breadcrumb-title-color']) ? $_POST['akd-framework-admin-page']['opt-breadcrumb-title-color'] : '';
			// $akd_theme_mods['breadcrumb_bg_color'] = isset($_POST['akd-framework-admin-page']['opt-breadcrumb-bg-color']) ? $_POST['akd-framework-admin-page']['opt-breadcrumb-bg-color'] : '';
			// $akd_theme_mods['hostiko_breadcrumb_color'] = isset($_POST['akd-framework-admin-page']['opt-breadcrumb-color']) ? $_POST['akd-framework-admin-page']['opt-breadcrumb-color'] : '';
			// $akd_theme_mods['hostiko_breadcrumb_current_page_color'] = isset($_POST['akd-framework-admin-page']['opt-breadcrumb-current-page-color']) ? $_POST['akd-framework-admin-page']['opt-breadcrumb-current-page-color'] : '';
			// $akd_theme_mods['hostiko_breadcrumb_hover_color'] = isset($_POST['akd-framework-admin-page']['opt-breadcrumb-hover-color']) ? $_POST['akd-framework-admin-page']['opt-breadcrumb-hover-color'] : '';
			// $akd_theme_mods['hostiko_breadcrumb_active_color'] = isset($_POST['akd-framework-admin-page']['opt-breadcrumb-active-color']) ? $_POST['akd-framework-admin-page']['opt-breadcrumb-active-color'] : '';
			// $akd_theme_mods['hostiko_breadcrumb_br'] = isset($_POST['akd-framework-admin-page']['opt-breadcrumb-border-radius']) ? $_POST['akd-framework-admin-page']['opt-breadcrumb-border-radius'] : '';
			// $akd_theme_mods['hostiko_breadcrumb_border_width'] = isset($_POST['akd-framework-admin-page']['opt-breadcrumb-border-width']) ? $_POST['akd-framework-admin-page']['opt-breadcrumb-border-width'] : '';
			// $akd_theme_mods['hostiko_breadcrumb_border_color'] = isset($_POST['akd-framework-admin-page']['opt-breadcrumb-border-color']) ? $_POST['akd-framework-admin-page']['opt-breadcrumb-border-color'] : '';
			// $akd_theme_mods['hostiko_breadcrumb_padding'] = isset($_POST['akd-framework-admin-page']['opt-breadcrumb-padding']) ? $_POST['akd-framework-admin-page']['opt-breadcrumb-padding'] : '';
			
			
			// $akd_theme_mods['sub-banner-img'] = isset($_POST['akd-framework-admin-page']['opt-subBannerImage']) ? $_POST['akd-framework-admin-page']['opt-subBannerImage'] : '';
			// $akd_theme_mods['breadcrumb_blog_title'] = isset($_POST['akd-framework-admin-page']['opt-blog-page-title']) ? $_POST['akd-framework-admin-page']['opt-blog-page-title'] : '';
			// $akd_theme_mods['breadcrumb_blog_title_details'] = isset($_POST['akd-framework-admin-page']['opt-blog-page-subTitle']) ? $_POST['akd-framework-admin-page']['opt-blog-page-subTitle'] : '';
			// $akd_theme_mods['hostiko_error_404_text'] = isset($_POST['akd-framework-admin-page']['opt-404-title']) ? $_POST['akd-framework-admin-page']['opt-404-title'] : '';
			// $akd_theme_mods['hostiko_error_title'] = isset($_POST['akd-framework-admin-page']['opt-404-not-found-title']) ? $_POST['akd-framework-admin-page']['opt-404-not-found-title'] : '';
			// $akd_theme_mods['hostiko_error_desc'] = isset($_POST['akd-framework-admin-page']['opt-404-desc']) ? $_POST['akd-framework-admin-page']['opt-404-desc'] : '';
			// $akd_theme_mods['hostiko_error_link_text'] = isset($_POST['akd-framework-admin-page']['opt-404-link-text']) ? $_POST['akd-framework-admin-page']['opt-404-link-text'] : '';
			// $akd_theme_mods['hostiko_preloader_switch'] = isset($_POST['akd-framework-admin-page']['opt-switcher-preloader']) ? $_POST['akd-framework-admin-page']['opt-switcher-preloader'] : '';
			// $akd_theme_mods['show_preloader'] = isset($_POST['akd-framework-admin-page']['opt-switcher-preloader']) ? $_POST['akd-framework-admin-page']['opt-switcher-preloader'] : '';
			
			// $akd_theme_mods['hostiko_show_investment_offer_link'] = isset($_POST['akd-framework-admin-page']['opt-switcher-showinvestment']) ? $_POST['akd-framework-admin-page']['opt-switcher-showinvestment'] : '';
			// $akd_theme_mods['hostiko_header_link_text'] = isset($_POST['akd-framework-admin-page']['investment-button-txt']) ? $_POST['akd-framework-admin-page']['investment-button-txt'] : '';
			// $akd_theme_mods['hostiko_header_link_url'] = isset($_POST['akd-framework-admin-page']['investment-btn-link']['url']) ? $_POST['akd-framework-admin-page']['investment-btn-link']['url'] : '';
			
			// $akd_theme_mods['hostiko_show_contact_info'] = isset($_POST['akd-framework-admin-page']['opt-switcher-show-contact-info']) ? $_POST['akd-framework-admin-page']['opt-switcher-show-contact-info'] : '';
			// $akd_theme_mods['hostiko_header_email'] = isset($_POST['akd-framework-admin-page']['email-txt']) ? $_POST['akd-framework-admin-page']['email-txt'] : '';
			// $akd_theme_mods['hostiko_header_phone'] = isset($_POST['akd-framework-admin-page']['phone-txt']) ? $_POST['akd-framework-admin-page']['phone-txt'] : '';
			
			
			// $akd_theme_mods['hostiko_enable_customizer_settings'] = isset($_POST['akd-framework-admin-page']['opt-switcher-customizer-settings']) ? $_POST['akd-framework-admin-page']['opt-switcher-customizer-settings'] : '';
			
			// $akd_theme_mods['hostiko_show_social_profiles'] = isset($_POST['akd-framework-admin-page']['opt-switcher-socialprofiles']) ? $_POST['akd-framework-admin-page']['opt-switcher-socialprofiles'] : '';
			// $akd_theme_mods['hostiko_topbar_fb_url'] = isset($_POST['akd-framework-admin-page']['social-fb-link']['url']) ? $_POST['akd-framework-admin-page']['social-fb-link']['url'] : '';
			// $akd_theme_mods['hostiko_topbar_twitter_url'] = isset($_POST['akd-framework-admin-page']['social-x-link']['url']) ? $_POST['akd-framework-admin-page']['social-x-link']['url'] : '';
			// $akd_theme_mods['hostiko_topbar_linkedin_url'] = isset($_POST['akd-framework-admin-page']['social-linkedin-link']['url']) ? $_POST['akd-framework-admin-page']['social-linkedin-link']['url'] : '';
			// $akd_theme_mods['hostiko_topbar_instagram_url'] = isset($_POST['akd-framework-admin-page']['social-instagram-link']['url']) ? $_POST['akd-framework-admin-page']['social-instagram-link']['url'] : '';

			// update_option( 'theme_mods_' . $theme_slug, $akd_theme_mods );
			
		}
	}

	/**
	 * Method to get elementor global site settings
	 */
	private function akd_get_elementor_global_settings() {
		/**
		 * Check if Elementor plugin is active
		 */
		if ( in_array( 'elementor/elementor.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ), true ) ) {
			if ( isset( $_POST['akd-framework-admin-page'] ) ) {

				if ( isset( $_POST['akd-framework-admin-page']['akd-theme-colors'] ) && is_array( $_POST['akd-framework-admin-page']['akd-theme-colors'] ) ) {
					
					global $wpdb;
					
					$active_kit_id = get_option( 'elementor_active_kit' );
					if ( $active_kit_id ) {
						$post_meta = get_post_meta( $active_kit_id, '_elementor_page_settings', true );

						if ( $post_meta ) {
							$akd_theme_system_colors = $_POST['akd-framework-admin-page']['akd-theme-colors'];
							$akd_theme_custom_colors = $_POST['akd-framework-admin-page']['akd-custom-colors'];
							$akd_theme_typography    = $_POST['akd-framework-admin-page'];
							 /**
							 * Typography update
							 */
							if (isset($akd_theme_typography)) {
								// echo 'post_meta: <pre>' . print_r($post_meta, true) . '</pre>';die;
								foreach ($post_meta['system_typography'] as &$typography) {
									// Extract the ID from the _id field
									$id = $typography['_id'];
									
									// Construct the dynamic keys for font family and weight
									$font_family_key = 'akd-' . $id . '-typography';
									$font_weight_key = 'akd-' . $id . '-font-weight';
									$font_font_size = 'akd-' . $id . '-font-size';
									$font_text_transform = 'akd-' . $id . '-text-transform';
							
									// Check if the corresponding typography settings exist and update
									if (isset($akd_theme_typography[$font_family_key])) {
										$typography['typography_font_family'] = $akd_theme_typography[$font_family_key];
									}
									if (isset($akd_theme_typography[$font_weight_key])) {
										$typography['typography_font_weight'] = $akd_theme_typography[$font_weight_key];
									}
									if (isset($akd_theme_typography[$font_font_size])) {
										$typography['typography_font_size']['size'] = $akd_theme_typography[$font_font_size];
									}
									if (isset($akd_theme_typography[$font_text_transform])) {
										$typography['typography_text_transform'] = $akd_theme_typography[$font_text_transform];
									}
								}
								foreach ($post_meta['custom_typography'] as &$typography) {
									$id = $typography['_id'];
									
									// Construct the dynamic keys for font family and weight
									$font_family_key = 'akd-' . $id . '-typography';
									$font_weight_key = 'akd-' . $id . '-font-weight';
									$font_font_size = 'akd-' . $id . '-font-size';
									$font_text_transform = 'akd-' . $id . '-text-transform';
							
									// Check if the corresponding typography settings exist and update
									if (isset($akd_theme_typography[$font_family_key])) {
										$typography['typography_font_family'] = $akd_theme_typography[$font_family_key];
									}
									if (isset($akd_theme_typography[$font_weight_key])) {
										$typography['typography_font_weight'] = $akd_theme_typography[$font_weight_key];
									}
									if (isset($akd_theme_typography[$font_font_size])) {
										$typography['typography_font_size']['size'] = $akd_theme_typography[$font_font_size];
									}
									if (isset($akd_theme_typography[$font_text_transform])) {
										$typography['typography_text_transform'] = $akd_theme_typography[$font_text_transform];
									}
								}
							}
							
							// if(isset($akd_theme_typography['akd-primary-typography']) && isset($akd_theme_typography['akd-secondary-typography']) && isset($akd_theme_typography['akd-secondary-font-weight'])){
							// 	$post_meta['system_typography'][0]['typography_font_family'] = $akd_theme_typography['akd-primary-typography'];
							// 	$post_meta['system_typography'][0]['typography_font_weight'] = $akd_theme_typography['akd-primary-font-weight'];
							// 	$post_meta['system_typography'][1]['typography_font_family'] = $akd_theme_typography['akd-secondary-typography'];
							// 	$post_meta['system_typography'][1]['typography_font_weight'] = $akd_theme_typography['akd-secondary-font-weight'];
							// 	$post_meta['system_typography'][2]['typography_font_family'] = $akd_theme_typography['akd-text-typography'];
							// 	$post_meta['system_typography'][2]['typography_font_weight'] = $akd_theme_typography['akd-text-font-weight'];
							// }
						
							/**
							 * Update colors based on second array
							 */
							foreach ( $post_meta[ 'custom_colors' ] as &$color ) {
								/**
								 * Extract the ID from the _id field
								 */
								$id = $color['_id'];

								/**
								 * Check if the corresponding custom color exists
								 */
								$custom_color_key = 'custom-color-' . $id;
								if ( array_key_exists( $custom_color_key, $akd_theme_custom_colors ) ) {
									/**
									 * Update the color value
									 */
									$color['color'] = $akd_theme_custom_colors[ $custom_color_key ];
								}
							}

							$post_meta['system_colors'][0]['color'] = $akd_theme_system_colors['primary'] ? $akd_theme_system_colors['primary'] : '';
							$post_meta['system_colors'][1]['color'] = $akd_theme_system_colors['secondary'] ? $akd_theme_system_colors['secondary'] : '';
							$post_meta['system_colors'][2]['color'] = $akd_theme_system_colors['text'] ? $akd_theme_system_colors['text'] : '';
							$post_meta['system_colors'][3]['color'] = $akd_theme_system_colors['accent'] ? $akd_theme_system_colors['accent'] : '';

							update_post_meta( $active_kit_id, '_elementor_page_settings', $post_meta );

							$active_kit = \Elementor\Plugin::$instance->kits_manager->get_active_kit_for_frontend();
							if ( $active_kit ) {

								$query = $wpdb->prepare(
									"
									SELECT *
									FROM $wpdb->postmeta
									WHERE 
										CONVERT(meta_id USING utf8) LIKE %s OR
										CONVERT(post_id USING utf8) LIKE %s OR
										CONVERT(meta_key USING utf8) LIKE %s OR
										CONVERT(meta_value USING utf8) LIKE %s
									",
									'%_elementor_page_settings%',
									'%_elementor_page_settings%',
									'%_elementor_page_settings%',
									'%_elementor_page_settings%'
								);

								$results = $wpdb->get_results( $query );

								/**
								 * Handle results
								 */
								if ( $results ) {
									foreach ( $results as $key => $result ) {
										if ( $key == 0 ) {
											$elementor_post_data = get_post_meta( $result->post_id, '_elementor_page_settings', true );
											$elementor_post_data['system_colors'][0]['color'] = $akd_theme_system_colors['primary'] ? $akd_theme_system_colors['primary'] : '';
											$elementor_post_data['system_colors'][1]['color'] = $akd_theme_system_colors['secondary'] ? $akd_theme_system_colors['secondary'] : '';
											$elementor_post_data['system_colors'][2]['color'] = $akd_theme_system_colors['text'] ? $akd_theme_system_colors['text'] : '';
											$elementor_post_data['system_colors'][3]['color'] = $akd_theme_system_colors['accent'] ? $akd_theme_system_colors['accent'] : '';

											$all_settings = $active_kit->get_settings();
											$all_settings['system_colors'][0]['color'] = $akd_theme_system_colors['primary'] ? $akd_theme_system_colors['primary'] : '';
											$all_settings['system_colors'][1]['color'] = $akd_theme_system_colors['secondary'] ? $akd_theme_system_colors['secondary'] : '';
											$all_settings['system_colors'][2]['color'] = $akd_theme_system_colors['text'] ? $akd_theme_system_colors['text'] : '';
											$all_settings['system_colors'][3]['color'] = $akd_theme_system_colors['accent'] ? $akd_theme_system_colors['accent'] : '';

											$active_kit->set_settings( $all_settings );
											$active_kit->save( $all_settings );

											$elementor_global_settings = json_encode( $active_kit->get_settings(), true );
											update_option( '_akd_elementor_global', $elementor_global_settings );
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}

	/**
	 * Method to update Elementor global site settings
	 *
	 * @param array $options An associative array of settings to update.
	 */
	public function akd_update_elementor_global_settings( $options ) {}

	/**
	 * Method to validate WHMCS URL
	 */
	public function akd_check_whmcs_url() {
		if ( isset( $_POST['action'] ) && 'akd_check_whmcs_url' === trim( $_POST['action'] ) ){

			if ( get_option( '_purchase_code_validate' ) == true ) {
				/**
				 *  Use the API credentials
				 */
				$api_url  = $this->akd_adjust_whmcs_url( trim( $_POST['url'] ) );
				$handles = curl_init( $api_url );
				curl_setopt( $handles, CURLOPT_NOBODY, true );
				curl_setopt( $handles, CURLOPT_TIMEOUT, 10 );
				curl_exec( $handles );
				$resultat = curl_getinfo( $handles, CURLINFO_HTTP_CODE );

				if ( $resultat ) {
						$_return['type'] = 'success';
						$_return['msg']  = 'SUCCESS: Curl Connected Successfully';
				} else {
					$_return['type'] = 'error';
					$_return['msg']  = 'Unable to connect to Server. Please contact to your hosting provider to enable your Curl';
				}
			} else {
				$_return['type'] = 'error';
				$_return['msg']  = 'Please activate your theme first.';
			}
		}

		echo json_encode( $_return );

		die();
	}

	/**
	 * Adjusts the WHMCS API URL as needed.
	 *
	 * @param string $api_url The original WHMCS API URL.
	 * @return string The adjusted WHMCS API URL.
	 */
	private function akd_adjust_whmcs_url( $api_url ) {
		$api_url = rtrim( $api_url, '/' );
		if ( ! preg_match( '/includes/', $api_url ) ) {
			$api_url .= '/includes';
		}

		if ( ! preg_match( '/api.php/', $api_url ) ) {
			$api_url .= '/api.php';
		}

		return $api_url;
	}

	/**
	 * Method to WHMCS groups
	 */
	public function akd_get_whmcs_groups() {
		$_return = array();
		if ( isset( $_POST['action'] ) && 'akd_get_whmcs_groups' === trim( $_POST['action'] ) ) {

			/**
			 * Use the API credentials
			 */
			$api_url        = $this->akd_adjust_whmcs_url( trim( $_POST['url'] ) );
			$api_secret     = trim( $_POST['secret'] );
			$api_identifier = trim( $_POST['identifier'] );

			$_return = self::akd_get_data(
				array(
					'url'        => $api_url,
					'identifier' => $api_identifier,
					'secret'     => $api_secret,
				),
			);
		}

		echo json_encode( $_return );
		die();
	}

	/**
	 * Retrieves data based on the provided configurations.
	 *
	 * @param array $configs The configurations used to retrieve the data.
	 * @return mixed The retrieved data, or false on failure.
	 */
	public function akd_get_data( $configs ) {
		$api_url        = $configs['url'];
		$api_identifier = $configs['identifier'];
		$api_secret     = $configs['secret'];

		/**
		 * Initialize the cURL resource
		 */
		$ch = $this->akd_init_curl( $api_url );

		/**
		 * Call WHMCS API to get all products
		 */
		$params = array(
			'identifier'   => $api_identifier,
			'secret'       => $api_secret,
			'action'       => 'GetProducts',
			'responsetype' => 'json',
		);

		$response_data = $this->akd_call_whmcs_api( $ch, $params );

		/**
		 * Close the cURL resource
		 */
		$this->akd_close_curl( $ch );

		$_return = array();
		$html    = '';
		if ( 'success' === $response_data['result'] ) {
			/**
			 * Save WHMCS data
			 */
			update_option( 'whmcs_url', $api_url );
			update_option( 'whmcs_secret', $this->encrypt_data( $api_secret, 'a3f9e1b2c5d8e0a7f6b9d2c4a7e9f2b0c5e8a0f5b7e9a3f2b0c5e8a0f5b7e9' ) );
			update_option( 'whmcs_identifier', $this->encrypt_data( $api_identifier, 'a3f9e1b2c5d8e0a7f6b9d2c4a7e9f2b0c5e8a0f5b7e9a3f2b0c5e8a0f5b7e9' ) );

			if ( isset( $response_data['products'] ) ) {
				update_option( 'whmcs_products_data', json_encode( $response_data ) );

				$_return['type'] = 'success';
				$_return['data'] = $response_data;
				$_return['msg']  = 'SUCCESS: WHMCS data synced successfully.';
			} else {
				$_return['type'] = 'success';
				$_return['data'] = $response_data;
				$_return['msg']  = 'SUCCESS: WHMCS data synced successfully, No products Found';
			}
		} else {
			$_return['type'] = 'error';
			$_return['data'] = $response_data;
			$_return['msg']  = 'ERROR: ' . $response_data['message'];
		}

		return $_return;
	}

	/**
	 * Initializes a cURL session for the given API URL.
	 *
	 * @param string $api_url The API URL to initialize the cURL session for.
	 * @return resource|false The initialized cURL handle, or false on failure.
	 */
	public function akd_init_curl( $api_url ) {
		$ch = curl_init();
		curl_setopt( $ch, CURLOPT_URL, $api_url );
		curl_setopt( $ch, CURLOPT_POST, true );
		curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
		return $ch;
	}

	/**
	 * Calls the WHMCS API using a cURL session.
	 *
	 * @param resource $ch     The cURL handle to use for the API call.
	 * @param array    $params The parameters to send to the WHMCS API.
	 * @return mixed           The response from the WHMCS API, or false on failure.
	 */
	public function akd_call_whmcs_api( $ch, $params ) {
		curl_setopt( $ch, CURLOPT_POSTFIELDS, http_build_query( $params ) );
		$response = curl_exec( $ch );
		if ( curl_errno( $ch ) ) {
			$error = curl_error( $ch );
			echo 'CURL Error: ' . $error;
		} else {
			$response_data = json_decode( $response, true );
			return $response_data;
		}
	}

	/**
	 * Closes a cURL session.
	 *
	 * @param resource $ch The cURL handle to close.
	 * @return void
	 */
	public function akd_close_curl( $ch ) {
		curl_close( $ch );
	}

	/**
	 * Encrypts the given data using the specified key.
	 *
	 * @param string $data The data to be encrypted.
	 * @param string $key  The key to use for encryption.
	 * @return string|false The encrypted data, or false on failure.
	 */
	public function encrypt_data( $data, $key ) {
		$cipher    = 'aes-256-cbc';
		$iv        = openssl_random_pseudo_bytes( openssl_cipher_iv_length( $cipher ) );
		$encrypted = openssl_encrypt( $data, $cipher, $key, 0, $iv );
		return base64_encode( $iv . $encrypted );
	}

	/**
	 * Method to get WHMCS TLDs
	 */
	public function akd_get_whmcs_tlds() {
		$api_url        = $this->akd_adjust_whmcs_url( trim( $_POST['url'] ) );
		$api_secret     = trim( $_POST['secret'] );
		$api_identifier = trim( $_POST['identifier'] );

		$ch = $this->akd_init_curl( $api_url );

		/**
		 * Call WHMCS API to get TLD pricing
		 */
		$params = array(
			'identifier'   => $api_identifier,
			'secret'       => $api_secret,
			'action'       => 'GetTLDPricing',
			'responsetype' => 'json',
		);

		$response_data = $this->akd_call_whmcs_api( $ch, $params );
		$this->akd_close_curl( $ch );
		$_return = array();
		if ( 'success' === $response_data['result'] ) {
			if ( isset( $response_data['pricing'] ) ) {
				update_option( 'whmcs_products_tld_data', json_encode( $response_data ) );
				$_return['type'] = 'success';
				$_return['data'] = json_encode( $response_data );
				$_return['msg']  = 'SUCCESS: WHMCS TLD data synced successfully.';
			}
		} else {
			$_return['type'] = 'error';
			$_return['data'] = json_encode( $response_data );
			$_return['msg']  = 'ERROR: ' . $response_data['message'];
		}

		echo json_encode( $_return );
		die();
	}

	/**
	 * Method to resync WHMCS
	 */
	public function akd_get_groups_resync() {
		$_return = array();
		if ( isset( $_POST['action'] ) && 'akd_get_groups_resync' === trim( $_POST['action'] ) ) {
			$api_url = $this->akd_adjust_whmcs_url( trim( $_POST['url'] ) );

			$temp_secret     = trim( $_POST['secret'] );
			$temp_identifier = trim( $_POST['identifier'] );

			$api_secret     = decrypt_data( $temp_secret, 'a3f9e1b2c5d8e0a7f6b9d2c4a7e9f2b0c5e8a0f5b7e9a3f2b0c5e8a0f5b7e9' );
			$api_identifier = decrypt_data( $temp_identifier , 'a3f9e1b2c5d8e0a7f6b9d2c4a7e9f2b0c5e8a0f5b7e9a3f2b0c5e8a0f5b7e9' );
			$api_secret     = $api_secret == "" ? $temp_secret : $api_secret;
			$api_identifier = $api_identifier == "" ? $temp_identifier : $api_identifier;

			$configs = array(
				'url'        => $api_url,
				'identifier' => $api_identifier,
				'secret'     => $api_secret,
			);

			$_return = self::akd_get_data( $configs );
		}

		echo json_encode( $_return );
		die();
	}

	/**
	 * Decrypts the given data using the specified key.
	 *
	 * @param string $data The encrypted data to be decrypted.
	 * @param string $key  The key to use for decryption.
	 * @return string|false The decrypted data, or false on failure.
	 */
	private function decrypt_data( $data, $key ) {
		$cipher    = 'aes-256-cbc';
		$data      = base64_decode( $data );
		$iv_length = openssl_cipher_iv_length( $cipher );
		$iv        = substr( $data, 0, $iv_length );
		$data      = substr( $data, $iv_length );
		return openssl_decrypt( $data, $cipher, $key, 0, $iv );
	}
	public function reset_wordpress_and_reactivate() {
		 // Ensure the user has the necessary permissions
		 if (!current_user_can('manage_options')) {
			wp_send_json_error('Unauthorized');
			return;
		}
	
		global $wpdb;
	
		// Step 1: Save current theme, plugins, and important options
		$current_theme = get_option('stylesheet');
		$active_plugins = get_option('active_plugins');
		
		// Save the specific options to preserve
		$preserved_options = array(
			'_purchase_code' => get_option('_purchase_code'),
			'_encryption_key' => get_option('_encryption_key'),
			'_purchase_code_validate' => get_option('_purchase_code_validate'),
			'blogname',       // Site Title
	 		'home',           // WP Address (URL)
	 		'siteurl',        // Site Address (URL)
	 		'blogdescription',// Tagline
	 		'permalink_structure', // Permalink Structure
	 		'timezone_string', // Timezone
	 		'admin_email',    // Admin Email
	 		'search_engine_visibility', // Search Engine Visibility
	 		'WPLANG',          // Site Language
			'_akd_api_server_url',
			'_akd_download_url',
		);
	
		// Step 2: Reset the wp_options table
		// Delete all rows except the preserved options
		$options_to_preserve = array('_purchase_code', '_encryption_key', '_purchase_code_validate', 'blogname','home','siteurl', 'blogdescription','permalink_structure','timezone_string', 'admin_email','search_engine_visibility','WPLANG');
		$placeholders = implode(',', array_fill(0, count($options_to_preserve), '%s'));
		
		delete_option('_akd_imported_layout');
		$wpdb->query($wpdb->prepare(
			"DELETE FROM {$wpdb->options} WHERE (option_name LIKE %s OR option_name LIKE %s) AND option_name NOT IN ($placeholders)", '_transient_%', '_site_transient_%', 'mods\_%', 'theme_mods\_%', ...$options_to_preserve
		));
	
		// Restore preserved options
		foreach ($preserved_options as $key => $value) {
			update_option($key, $value);
		}

		// Step 3: Manually reset the database content (posts, pages, etc.)
		// Delete all posts, pages, custom post types, etc.
		$wpdb->query("DELETE FROM {$wpdb->posts} WHERE post_type IN ('post', 'page', 'attachment', 'custom_post_type')");
	
		// Delete all post meta
		// $wpdb->query("DELETE FROM {$wpdb->postmeta} WHERE meta_key != '_elementor_page_settings'");
		// $wpdb->query("DELETE FROM {$wpdb->postmeta}");
	
		// Delete all terms, term relationships, and term taxonomies
		$wpdb->query("DELETE FROM {$wpdb->terms}");
		$wpdb->query("DELETE FROM {$wpdb->term_relationships}");
		$wpdb->query("DELETE FROM {$wpdb->term_taxonomy}");
	
		// Delete all comments
		$wpdb->query("DELETE FROM {$wpdb->comments}");
		$wpdb->query("DELETE FROM {$wpdb->commentmeta}");
	
		// // Step 4: Reset user roles and capabilities manually
		// remove_all_filters('wp_roles_init');
		// wp_roles()->for_site(get_current_blog_id());
		// wp_roles()->reset();
	
		// Step 5: Reactivate saved theme and plugins
		switch_theme($current_theme);
		foreach ($active_plugins as $plugin) {
			activate_plugin($plugin);
		}
	
		
		// Optional: Clean the cache if any
		wp_cache_flush();
	
		wp_send_json_success('WordPress reset successfully');
	}

	// Validate WHMCS URL 
	
	public function akd_validate_whmcs_url() {
		// Retrieve the URL from POST data
		$url = $_POST['url'];
		// Check if the URL is properly formatted
		if (!filter_var($url, FILTER_VALIDATE_URL)) {
			$_return['type'] = 'error';
			$_return['msg'] = 'Please enter a valid URL';
		} else {
			$_return['type'] = 'success';
			$_return['msg'] = 'SUCCESS: URL is valid';
		}
	
		// Return the result as a JSON response
		echo json_encode($_return);
		die();
	}

	
}