<?php
/**
 * Abstract Class for creating meta boxes using CodeStart Framework
 *
 * @since      0.9.0
 * @package    Akd_Framework
 * @subpackage Akd_Framework/includes
 */
abstract class Akd_Framework_Meta {
	/**
	 * The prefix used for the plugin.
	 *
	 * @var string
	 */
	private $prefix;

	/**
	 * The custom post type associated with the plugin.
	 *
	 * @var string
	 */
	protected $post_type;

	/**
	 * The prefix used for member variables.
	 *
	 * @var string
	 */
	private $mem_prefix;


	/**
	 * Constructor for the class.
	 *
	 * @param string $plugin_name The name or prefix of the plugin.
	 */
	public function __construct( $plugin_name ) {
		$this->prefix = $plugin_name;
		$this->init_meta();
		$this->introduce_tabs();
		$this->admin_init();
		
	}

	/**
	 * Method to create admin options
	 *
	 * @return void
	 */
	private function admin_init() {
		$activated_theme = wp_get_theme();
		$theme_name      = explode( ' ', $activated_theme->get( 'Name' ) )[0];
		$args = array(
			'post_type' => 'elementor_library',
			'post_status' => 'publish',
			'posts_per_page' => -1,
	
		);
		$hostiko_the_query = new WP_Query($args);
		$hostiko_elementor_posts = array();
		if ($hostiko_the_query->have_posts()):
			foreach ($hostiko_the_query->posts as $hostiko_post):
				$hostiko_elementor_posts[$hostiko_post->ID] = apply_filters('the_title', get_the_title($hostiko_post));
			endforeach;
		endif;
		$hostiko_elementor_template_list = $hostiko_elementor_posts;
		/**
		 * Create options
		 */
		CSF::createOptions(
			'akd-framework-admin-page',
			array(
				'menu_title'         => esc_html( $theme_name . ' Options' ),
				'menu_slug'          => 'akd-framework',
				'framework_title'    => esc_html( $theme_name . ' Options' ),
				'menu_position'      => 15,
				'show_search'        => false,
				'show_reset_all'     => false,
				'show_reset_section' => false,
				'ajax_save'          => false,
				'menu_icon'          => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/akd-demo-importer-icon.png',
			)
		);

		/**
		 * Theme activation
		 */
		CSF::createSection(
			'akd-framework-admin-page',
			array(
				'title'  => esc_html( 'Dashboard' ),
				'icon'   => 'fas fa-tachometer-alt',
				'fields' => array(
					array(
						'id'    => 'akd-theme-dashboard',
						'type'  => 'tabbed',
						'class' => 'akd-framework-tabbed',
						'tabs'  => array(
							array(
								'title'  => 'Theme Activation',
								'icon'   => 'fa fa-cube',
								'fields' => array(
									array(
										'id'          => 'akd-act-name',
										'type'        => 'text',
										'title'       => esc_html( 'Full Name *' ),
										'placeholder' => esc_html( 'Your name.' ),
										'class'       => 'akd-full-name',
										'required'    => true,
									),

									array(
										'id'          => 'akd-act-email',
										'type'        => 'text',
										'title'       => esc_html( 'Email *' ),
										'placeholder' => esc_html( 'Your email.' ),
										'class'       => 'akd-email',
										'required'    => true,
									),

									array(
										'id'      => 'akd-act-options',
										'type'    => 'radio',
										'title'   => esc_html( 'Activation Method' ),
										'options' => array(
											'purchase-code' 	=> esc_html( 'Already have a purchase code' ),
											'get-purchase-code' => esc_html( 'Downloaded Hostiko from Envato Elements' ),
										),
										'default' => 'purchase-code',
										'class'   => 'akd-act-option',
									),
								),
							),
							array(
								'title'  => esc_html( 'Plugins' ),
								'icon'   => 'fa fa-plug',
								'fields' => array(
									array(
										'type'     => 'callback',
										'function' => array( $this, 'akd_required_plugins_callback' ),
									),
								),
							),
							array(
								'title'  => esc_html( 'Demo Data' ),
								'icon'   => 'fa fa-desktop',
								'fields' => array(
									array(
										'type'     => 'callback',
										'function' => array( $this, 'akd_theme_demo_importer_callback' ),
									),
								),
							),

							array(
								'title'  => esc_html( 'Install Demo' ),
								'icon'   => 'fa fa-download',
								'fields' => array(
									array(
										'type'     => 'callback',
										'function' => array( $this, 'akd_install_demo_data_callback' ),
									),
								),
							),

							// $this->akd_get_elementor_theme_settings(),

						),
					),
				),
			)
		);

		CSF::createSection(
			'akd-framework-admin-page',
			array(
				'title'  => esc_html( 'Branding' ),
				'icon'   => 'fas fa-swatchbook',
				'fields' => array(
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Site Identity' ),
						'class'   => 'akd-support-section-title akd-title-options',
					),
				
					array(
						'id'           => 'akd_update_logo',
						'type'         => 'upload',
						'title'        => 'Choose Logo',
						'library'      => 'image',
						'placeholder'  => $this -> akd_get_theme_logo(),
						'button_title' => 'Upload',
						'remove_title' => 'Remove',
						'preview' => true,
						'default'      => $this -> akd_get_theme_logo(),
						'value'      => $this -> akd_get_theme_logo(),
					),
					
					  
					array(
						'id'           => 'akd_update_favicon',
						'type'         => 'upload',
						'title'        => 'Choose Favicon',
						'library'      => 'image',
						'button_title' => 'Upload',
						'default'      => $this -> akd_get_theme_favicon(),
						'remove_title' => 'Remove',
						'placeholder'  => $this -> akd_get_theme_favicon(),
						'preview' => true,
					  ),
					  
				),
			),
		);

		CSF::createSection(
			'akd-framework-admin-page',
			$this->akd_get_elementor_theme_typography_settings(),
		);

		CSF::createSection(
			'akd-framework-admin-page',
			$this->akd_get_elementor_theme_color_settings(),
			
		);

		CSF::createSection(
			'akd-framework-admin-page',
			array(
				'title'  => esc_html( 'Banner' ),
				'icon'   => 'fas fa-th-large',
				'fields' => array(
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Banner' ),
						'class'   => 'akd-support-section-title akd-title-options',
					),
					array(
						'id'      => 'opt-switcher-subbanner',
						'type'    => 'switcher',
						'title'   => 'Show Sub Banner',
						'default' => true
					),
					array(
						'id'    => 'opt-subBannerImage',
						'type'  => 'upload',
						'title' => 'Choose Image',
						'library'      => 'image',
						'button_title' => 'Upload',
						'remove_title' => 'Remove',
						'preview' => true,
						'dependency' => array( 'opt-switcher-subbanner', '==', 'true' ) 
					),	

					array(
						'id'    => 'opt-subbanner-padding-desktop',
						'type'  => 'text',
						'title' => 'Padding Desktop',
						'default' => '4px',
						'dependency' => array( 'opt-switcher-subbanner', '==', 'true' ) 
					),	
					
					array(
						'id'    => 'opt-subbanner-padding-laptop',
						'type'  => 'text',
						'title' => 'Padding Laptop',
						'default' => '4px',
						'dependency' => array( 'opt-switcher-subbanner', '==', 'true' ) 
					),	
					array(
						'id'    => 'opt-subbanner-padding-tablet',
						'type'  => 'text',
						'title' => 'Padding Tablet',
						'default' => '4px',
						'dependency' => array( 'opt-switcher-subbanner', '==', 'true' ) 
					),	
					array(
						'id'    => 'opt-subbanner-padding-mobile',
						'type'  => 'text',
						'title' => 'Padding Mobile',
						'default' => '4px',
						'dependency' => array( 'opt-switcher-subbanner', '==', 'true' ) 
					),
					
					
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Page Title Setting' ),
						'class'   => 'akd-support-section-title',
					),
					array(
						'id'      => 'opt-switcher-breadcrumb',
						'type'    => 'switcher',
						'title'   => 'Show Breadcrumb',
						'default' => true
					),
					
					array(
						'id'    => 'opt-breadcrumb-title-color',
						'type'  => 'color',
						'title' => 'Title Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-breadcrumb', '==', 'true' )
					),
					array(
						'id'    => 'opt-subtitle-color',
						'type'  => 'color',
						'title' => 'Subtitle Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-breadcrumb', '==', 'true' )
					),
					array(
						'id'    => 'opt-breadcrumb-bg-color',
						'type'  => 'color',
						'title' => 'Breadcrumb BG Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-breadcrumb', '==', 'true' )
					),
					array(
						'id'    => 'opt-breadcrumb-color',
						'type'  => 'color',
						'title' => 'Breadcrumb Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-breadcrumb', '==', 'true' )
					),
					array(
						'id'    => 'opt-breadcrumb-current-page-color',
						'type'  => 'color',
						'title' => 'Breadcrumb Current Page Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-breadcrumb', '==', 'true' )
					),
					array(
						'id'    => 'opt-breadcrumb-hover-color',
						'type'  => 'color',
						'title' => 'Breadcrumb Hover Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-breadcrumb', '==', 'true' )
					),
					
					array(
						'id'    => 'opt-breadcrumb-border-radius',
						'type'  => 'text',
						'title' => 'Breadcrumb Border Radius',
						'default' => '4px',
						'dependency' => array( 'opt-switcher-breadcrumb', '==', 'true' )
					),
					array(
						'id'    => 'opt-breadcrumb-border-width',
						'type'  => 'text',
						'title' => 'Breadcrumb Border Width',
						'default' => '4px',
						'dependency' => array( 'opt-switcher-breadcrumb', '==', 'true' )
					),
					array(
						'id'    => 'opt-breadcrumb-border-color',
						'type'  => 'color',
						'title' => 'Breadcrumb Border Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-breadcrumb', '==', 'true' )
					),
					array(
						'id'    => 'opt-breadcrumb-padding',
						'type'  => 'text',
						'title' => 'Breadcrumb Padding',
						'default' => '12px 20px',
						'dependency' => array( 'opt-switcher-breadcrumb', '==', 'true' )
					),
				),
			),
		);
		
		CSF::createSection(
			'akd-framework-admin-page',
			array(
				'title'  => esc_html( 'Header' ),
				'icon'   => 'fas fa-grip-lines-vertical',
				'fields' => array(
					
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Header' ),
						'class'   => 'akd-support-section-title',
						'class'   => 'akd-support-section-title akd-title-options',
					),
					array(
						'id'          => 'opt-select-default-header',
						'type'        => 'select',
						'title'       => 'Choose Default Header',
						'placeholder' => 'Select an option',
						'options'     => array(
							'default-header' => esc_html( 'Default Header', 'resellpress' ),
							'elementor-header' => esc_html( 'Elementor Header', 'resellpress' ),
						),
						'default'     => 'default-header'
					),
					array(
						'id'          => 'opt-select-header-elementor-template',
						'type'        => 'select',
						'title'       => 'Header Elementor Templates',
						'placeholder' => 'Select an option',
						'options'     => $hostiko_elementor_template_list,
						'default'     => 'option-1',
						'dependency' => array( 'opt-select-default-header', '==', 'elementor-header' ),
					),
					array(
						'id'      => 'opt-switcher-stikyheader',
						'type'    => 'switcher',
						'title'   => 'Enable Sticky Header',
						'default' => false,
						'dependency' => array( 'opt-select-default-header', '==', 'default-header' ),
					),
					array(
						'id'    => 'opt-switcher-stikyheader-bg',
						'type'  => 'color',
						'title' => 'Stickyheader Background Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-select-default-header', '==', 'default-header' )
					),	
					
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Preloader' ),
						'class'   => 'akd-support-section-title',
					),
					array(
						'id'      => 'opt-switcher-preloader',
						'type'    => 'switcher',
						'title'   => 'Enable Preloader',
						'default' => false
					),
					array(
						'id'          => 'opt-select-preloader-img',
						'type'        => 'upload',
						'title'       => 'Preloader Image',
						'library'      => 'image',
						'button_title' => 'Upload',
						'remove_title' => 'Remove',
						'preview' => true,
						'dependency' => array( 'opt-switcher-preloader', '==', 'true' ),
					),
					array(
						'id'    => 'opt-switcher-preloader-bg',
						'type'  => 'color',
						'title' => 'Preloader Background Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-preloader', '==', 'true' )
					),	


					array(
						'type'    => 'heading',
						'content' => esc_html( 'Top Bar' ),
						'class'   => 'akd-support-section-title',
					),
					array(
						'id'      => 'opt-switcher-topbar',
						'type'    => 'switcher',
						'title'   => 'Enable Topbar',
						'default' => false
					),	
					array(
						'id'    => 'opt-switcher-topbar-bg',
						'type'  => 'color',
						'title' => 'Top Bar Background Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-topbar', '==', 'true' )
					),		
					array(
						'type'    => 'subheading',
						'content' => esc_html( 'Right Buttons' ),
						'class'   => 'akd-support-section-title',
					),		  
					array(
						'id'      => 'opt-switcher-showbutton',
						'type'    => 'switcher',
						'title'   => 'Show login button',
						'default' => false
					),	
					array(
						'id'      => 'login-button-txt',
						'type'    => 'text',
						'title'   => 'Text',
						'default' => 'Sign in',
						'dependency' => array( 'opt-switcher-showbutton', '==', 'true' )
					),
					array(
						'id'       => 'login-btn-link',
						'type'     => 'link',
						'title'    => 'Login In Link',
						'dependency' => array( 'opt-switcher-showbutton', '==', 'true' ),
					),
					array(
						'id'    => 'login-btn-icon',
						'type'  => 'icon',
						'title' => 'Login Button Icon',
						'dependency' => array( 'opt-switcher-showbutton', '==', 'true' ),
					),
					
					array(
						'id'      => 'opt-switcher-showinvestment',
						'type'    => 'switcher',
						'title'   => 'Show Investment Offer Link',
						'default' => false
					),	
					array(
						'id'      => 'investment-button-txt',
						'type'    => 'text',
						'title'   => 'Investment Link Text',
						'default' => 'Invest',
						'dependency' => array( 'opt-switcher-showinvestment', '==', 'true' )
					),
					array(
						'id'       => 'investment-btn-link',
						'type'     => 'link',
						'title'    => 'Investment Link URL',
						'dependency' => array( 'opt-switcher-showinvestment', '==', 'true' ),
					),

					array(
						'id'      => 'opt-switcher-socialprofiles',
						'type'    => 'switcher',
						'title'   => 'Show Social Profiles',
						'default' => false
					),
					array(
						'id'       => 'social-fb-link',
						'type'     => 'link',
						'title'    => 'Facebook URL',
						'default'  => '#',
						'dependency' => array( 'opt-switcher-socialprofiles', '==', 'true' ),
					),
					array(
						'id'       => 'social-x-link',
						'type'     => 'link',
						'title'    => 'X URL',
						'default'  => '#',
						'dependency' => array( 'opt-switcher-socialprofiles', '==', 'true' ),
					),
					array(
						'id'       => 'social-linkedin-link',
						'type'     => 'link',
						'title'    => 'LinkedIn URL',
						'default'  => '#',
						'dependency' => array( 'opt-switcher-socialprofiles', '==', 'true' ),
					),
					array(
						'id'       => 'social-instagram-link',
						'type'     => 'link',
						'title'    => 'Instagram URL',
						'default'  => '#',
						'dependency' => array( 'opt-switcher-socialprofiles', '==', 'true' ),
					),

					array(
						'type'    => 'subheading',
						'content' => esc_html( 'Left Buttons' ),
						'class'   => 'akd-support-section-title',
					),
					array(
						'id'      => 'opt-switcher-show-contact-info',
						'type'    => 'switcher',
						'title'   => 'Show Contact Info',
						'default' => false
					),	
					array(
						'id'      => 'email-txt',
						'type'    => 'text',
						'title'   => 'Email Address',
						'default' => 'info@gmail.com',
						'dependency' => array( 'opt-switcher-show-contact-info', '==', 'true' )
					),
					array(
						'id'      => 'phone-txt',
						'type'    => 'text',
						'title'   => 'Phone Number',
						'default' => '+97657945737',
						'dependency' => array( 'opt-switcher-show-contact-info', '==', 'true' )
					),


					array(
						'type'    => 'heading',
						'content' => esc_html( 'Navbar Style' ),
						'class'   => 'akd-support-section-title',
					),
					array(
						'id'    => 'opt-navbar-color',
						'type'  => 'color',
						'title' => 'Color',
						'default' => '#007bff',
					),	
					array(
						'id'    => 'opt-navbar-hover-color',
						'type'  => 'color',
						'title' => 'Hover Color',
						'default' => '#007bff',
					),	
					array(
						'id'    => 'opt-navbar-active-color',
						'type'  => 'color',
						'title' => 'Active Color',
						'default' => '#007bff',
					),	
					array(
						'id'    => 'opt-navbar-dropdown-bg',
						'type'  => 'color',
						'title' => 'Dropdown Background Color',
						'default' => '#007bff',
					),	
					array(
						'id'    => 'opt-navbar-dropdown-hover-bg',
						'type'  => 'color',
						'title' => 'Dropdown Hover background',
						'default' => '#007bff',
					),	
					array(
						'id'    => 'opt-navbar-dropdown-color',
						'type'  => 'color',
						'title' => 'Dropdown Color',
						'default' => '#007bff',
					),	
					array(
						'id'    => 'opt-navbar-dropdown-hover-color',
						'type'  => 'color',
						'title' => 'Dropdown Hover Color',
						'default' => '#007bff',
					),	
					array(
						'id'    => 'opt-navbar-dropdown-position',
						'type'  => 'text',
						'title' => 'Dropdown Position',
						'default' => '30px',
					),	
					array(
						'id'    => 'opt-navbar-dropdown-border-radius',
						'type'  => 'text',
						'title' => 'Dropdown Border Radius',
						'default' => '4px',
					),	
					array(
						'id'    => 'opt-navbar-dropdown-border-width',
						'type'  => 'text',
						'title' => 'Dropdown Border Width',
						'default' => '4px',
					),	
					array(
						'id'    => 'opt-navbar-dropdown-border-color',
						'type'  => 'color',
						'title' => 'Dropdown Border Color',
						'default' => '#007bff',
					),
					array(
						'id'    => 'opt-navbar-dropdown-bg-color-mobile',
						'type'  => 'color',
						'title' => 'Dropdown Backgound Color Mobile',
						'default' => '#007bff',
					),
					array(
						'id'    => 'opt-navbar-dropdown-color-mobile',
						'type'  => 'color',
						'title' => 'Dropdown Color Mobile',
						'default' => '#007bff',
					),
					array(
						'id'    => 'opt-navbar-dropdown-color-hover-mobile',
						'type'  => 'color',
						'title' => 'Dropdown Color Hover Mobile',
						'default' => '#007bff',
					),
					array(
						'id'    => 'opt-navbar-hamburger-color',
						'type'  => 'color',
						'title' => 'Hamburger Color',
						'default' => '#007bff',
					),
					array(
						'id'    => 'opt-navbar-dropdown-border-bottom-width',
						'type'  => 'text',
						'title' => 'Border Bottom Width',
						'default' => '0px',
					),	
					array(
						'id'    => 'opt-navbar-border-bottom-color',
						'type'  => 'color',
						'title' => 'Border Bottom Color',
						'default' => '#007bff',
					),

					array(
						'type'    => 'heading',
						'content' => esc_html( 'Live Chat' ),
						'class'   => 'akd-support-section-title',
					),
					array(
						'id'      => 'opt-switcher-livechat',
						'type'    => 'switcher',
						'title'   => 'Apply Style',
						'default' => false
					),
					array(
						'id'    => 'opt-livechat-bg',
						'type'  => 'color',
						'title' => 'Background Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-livechat', '==', 'true' ),
					),
					array(
						'id'    => 'opt-livechat-color',
						'type'  => 'color',
						'title' => 'Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-livechat', '==', 'true' ),
					),
					array(
						'id'    => 'opt-livechat-border-color',
						'type'  => 'color',
						'title' => 'Border Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-livechat', '==', 'true' ),
					),
					array(
						'id'    => 'opt-livechat-border-radius',
						'type'  => 'text',
						'title' => 'Border Radius',
						'default' => '4px',
						'dependency' => array( 'opt-switcher-livechat', '==', 'true' ),
					),
					array(
						'id'    => 'opt-livechat-padding',
						'type'  => 'text',
						'title' => 'Padding',
						'default' => '10px 10px 10px 10px',
						'dependency' => array( 'opt-switcher-livechat', '==', 'true' ),
					),
					array(
						'id'    => 'opt-livechat-border-hover-color',
						'type'  => 'color',
						'title' => 'Border Hover Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-livechat', '==', 'true' ),
					),
					array(
						'id'    => 'opt-livechat-bg-hover-color',
						'type'  => 'color',
						'title' => 'Background Hover Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-livechat', '==', 'true' ),
					),
					array(
						'id'    => 'opt-livechat-hover-color',
						'type'  => 'color',
						'title' => 'Hover Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-livechat', '==', 'true' ),
					),
					
				),
			),
		);
		
		CSF::createSection(
			'akd-framework-admin-page',
			array(
				'title'  => esc_html( 'Footer' ),
				'icon'   => 'fas fa-grip-lines',
				'fields' => array(
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Footer' ),
						'class'   => 'akd-support-section-title akd-title-options',
					),
					
					array(
						'id'          => 'opt-select-default-footer',
						'type'        => 'select',
						'title'       => 'Choose Default Footer',
						'placeholder' => 'Select an option',
						'options'     => array(
							'default-footer' => esc_html( 'Default Footer', 'hostiko' ),
							'elementor-footer' => esc_html( 'Elementor Footer', 'hostiko' ),
						),
						'default'     => 'default-footer'
					),
					array(
						'id'          => 'opt-select-footer-elementor-template',
						'type'        => 'select',
						'title'       => 'Footer Elementor Templates',
						'placeholder' => 'Select an option',
						'options'     => $hostiko_elementor_template_list,
						'default'     => 'option-1',
						'dependency' => array( 'opt-select-default-footer', '==', 'elementor-footer' ),
					),
					array(
						'id'      => 'copyright-txt',
						'type'    => 'text',
						'title'   => 'Copyright Text',
						'default' => 'Copyright © Hostiko 2025. All rights reserved',
						'dependency' => array( 'opt-select-default-footer', '==', 'default-footer' ),
					),
					
				),
			),
		);
		
		CSF::createSection(
			'akd-framework-admin-page',
			array(
				'title'  => esc_html( 'Blog/Archive' ),
				'icon'   => 'fas fa-newspaper',
				'fields' => array(
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Blog/Archive' ),
						'class'   => 'akd-support-section-title akd-title-options',
					),
					array(
						'id'          => 'opt-select-blog-elementor-template',
						'type'        => 'select',
						'title'       => 'Blog Page Elementor Template',
						'placeholder' => 'Select an option',
						'options'     => $hostiko_elementor_template_list,
						'default'     => 'option-1',
					),
					array(
						'id'      => 'opt-blog-page-title',
						'type'    => 'text',
						'title'   => 'Blog Page Title',
						'default' => 'Blog',
						'dependency' => array( 'opt-select-blog-elementor-template', '==', '' ),
					),
					array(
						'id'      => 'opt-blog-page-subTitle',
						'type'    => 'text',
						'title'   => 'Blog Page Sub Title',
						'default' => 'Read latest news from our blog & learn new things.',
						'dependency' => array( 'opt-select-blog-elementor-template', '==', '' ),
					),
					array(
						'id'        => 'opt-image-select-blog-layout',
						'type'      => 'image_select',
						'title'     => 'Blog Layout',
						'options'   => array(
							'left-sidebar' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/01.svg',
							'full-width' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/03.svg',
							'right-sidebar' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/02.svg',
						),
						'default'   => 'right-sidebar',
						'dependency' => array( 'opt-select-blog-elementor-template', '==', '' ),
					),
					array(
						'id'        => 'opt-image-select-blog-grid',
						'type'      => 'image_select',
						'title'     => 'Blog Grid',
						'options'   => array(
						  'single' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/06.svg',
						  'two-column' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/05.svg',
						  'three-column' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/04.svg',
						),
						'default'   => 'single',
						'dependency' => array( 'opt-select-blog-elementor-template', '==', '' ),
					),
					array(
						'id'    => 'opt-switcher-show-excerpt',
						'type'  => 'switcher',
						'title' => 'Show Excerpt',
						'default' => true,
						'dependency' => array( 'opt-select-blog-elementor-template', '==', '' ),
					),
					array(
						'id'    => 'opt-switcher-show-readmore',
						'type'  => 'switcher',
						'title' => 'Show Read More',
						'default' => true,
						'dependency' => array( 'opt-select-blog-elementor-template', '==', '' ),
					),
					array(
						'id'    => 'opt-switcher-show-date',
						'type'  => 'switcher',
						'title' => 'Show Date',
						'default' => true,
						'dependency' => array( 'opt-select-blog-elementor-template', '==', '' ),
					),
					

					// Single Blog Page
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Single Blog Page' ),
						'class'   => 'akd-support-section-title',
					),

					array(
						'id'          => 'opt-select-single-post-elementor-template',
						'type'        => 'select',
						'title'       => 'Single Post Elementor Template',
						'placeholder' => 'Select an option',
						'options'     => $hostiko_elementor_template_list,
						'default'     => 'option-1',
					),
					
					array(
						'id'        => 'opt-image-select-single-page-layout',
						'type'      => 'image_select',
						'title'     => 'Single Page Layout',
						'options'   => array(
							'left-sidebar' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/01.svg',
							'full-width' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/03.svg',
							'right-sidebar' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/02.svg',
						),
						'default'   => 'right-sidebar',
						'dependency' => array( 'opt-select-single-post-elementor-template', '==', '' ),
					),
				),
			),
		);
		
		CSF::createSection(
			'akd-framework-admin-page',
			array(
				'title'  => esc_html( 'Shop' ),
				'icon'   => 'fas fa-store',
				'fields' => array(
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Shop' ),
						'class'   => 'akd-support-section-title akd-title-options',
					),
					array(
						'id'      => 'opt-shop-title',
						'type'    => 'text',
						'title'   => 'Shop Page Title',
						'default' => 'Shop'
					),
					array(
						'id'      => 'opt-blog-shop-subTitle',
						'type'    => 'text',
						'title'   => 'Shop Page Sub Title',
						'default' => 'Check latest products from our store.'
					),
					array(
						'id'        => 'opt-image-select-shop-layout',
						'type'      => 'image_select',
						'title'     => 'Shop Layout',
						'options'   => array(
						  	'left-sidebar' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/01.svg',
							'full-width' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/03.svg',
							'right-sidebar' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/02.svg',
						),
						'default'   => 'full-width'
					),
					array(
						'id'        => 'opt-image-select-shop-grid',
						'type'      => 'image_select',
						'title'     => 'Shop Grid',
						'options'   => array(
						  'single' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/06.svg',
						  'two-column' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/05.svg',
						  'three-column' => plugin_dir_url( __FILE__ ) . 'codestar-framework/assets/images/04.svg',
						),
						'default'   => 'three-column'
					),
					
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Cart Icon' ),
						'class'   => 'akd-support-section-title',
					),
					array(
						'id'      => 'opt-switcher-carticon',
						'type'    => 'switcher',
						'title'   => 'Show Cart Icon',
						'default' => true
					),
					array(
						'id'    => 'opt-carticon-bg',
						'type'  => 'color',
						'title' => 'Background Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-carticon', '==', 'true' ),
					),
					array(
						'id'    => 'opt-carticon-color',
						'type'  => 'color',
						'title' => 'Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-carticon', '==', 'true' ),
					),
					array(
						'id'    => 'opt-carticon-border-color',
						'type'  => 'color',
						'title' => 'Border Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-carticon', '==', 'true' ),
					),
					array(
						'id'    => 'opt-carticon-border-radius',
						'type'  => 'text',
						'title' => 'Border Radius',
						'default' => '4px',
						'dependency' => array( 'opt-switcher-carticon', '==', 'true' ),
					),
					array(
						'id'    => 'opt-carticon-padding',
						'type'  => 'text',
						'title' => 'Padding',
						'default' => '10px 10px 10px 10px',
						'dependency' => array( 'opt-switcher-carticon', '==', 'true' ),
					),
					array(
						'id'    => 'opt-carticon-border-hover-color',
						'type'  => 'color',
						'title' => 'Border Hover Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-carticon', '==', 'true' ),
					),
					array(
						'id'    => 'opt-carticon-bg-hover-color',
						'type'  => 'color',
						'title' => 'Background Hover Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-carticon', '==', 'true' ),
					),
					array(
						'id'    => 'opt-carticon-hover-color',
						'type'  => 'color',
						'title' => 'Hover Color',
						'default' => '#007bff',
						'dependency' => array( 'opt-switcher-carticon', '==', 'true' ),
					),

					array(
						'type'    => 'heading',
						'content' => esc_html( 'Cart Counter' ),
						'class'   => 'akd-support-section-title',
					),
					array(
						'id'    => 'opt-cartcounter-bg-color',
						'type'  => 'color',
						'title' => 'Background Color',
						'default' => '#007bff',
					),
					array(
						'id'    => 'opt-cartcounter-color',
						'type'  => 'color',
						'title' => 'Color',
						'default' => '#007bff',
					),
					array(
						'id'    => 'opt-cartcounter-top',
						'type'  => 'text',
						'title' => 'Top',
						'default' => '0px',
					),
					array(
						'id'    => 'opt-cartcounter-right',
						'type'  => 'text',
						'title' => 'Right',
						'default' => '0px',
					),
				),
			),
		);

		if ( 'Hostiko' === $theme_name ) {
			/**
			 * Whmcs
			 */
			
			CSF::createSection(
				'akd-framework-admin-page',
				array(
					'title'  => esc_html( 'Integrations' ),
					'icon'   => 'fas fa-sitemap',
					'class'	=> 'akd-whmcs-api-integration',
					'fields' => array(
						
						array(
							'type'    => 'heading',
							'content' => esc_html( 'WHMCS API Credentials' ),
							'akd-support-section-title akd-title-options',
						),
						array(
							'id'    => 'akd-whmcs-api',
							'type'  => 'text',
							'title' => esc_html( 'WHMCS URL' ),
							'desc'  =>  '<i class="fas fa-info-circle"></i> How to find the WHMCS API URL. Visit <a href="https://akdesigner.com/whmcs-templates/index.php?rp=/knowledgebase/53/Configure-WHMCS-API-Key-for-Hostiko.html#whmcs-system-url" target="_blank">WHMCS URL</a>.' ,
						),

						array(
							'id'    => 'akd-whmcs-api-identifier',
							'type'  => 'text',
							'title' => esc_html( 'Identifier' ),
							'desc'  =>  '<i class="fas fa-info-circle"></i> How to find the WHMCS API Indentifier. Visit <a href="https://akdesigner.com/whmcs-templates/index.php?rp=/knowledgebase/53/Configure-WHMCS-API-Key-for-Hostiko.html#whmcs-api-credentials" target="_blank">API Indentifier</a>.',
						),

						array(
							'id'    => 'akd-whmcs-api-secret',
							'type'  => 'text',
							'title' => esc_html( 'Secret' ),
							'desc'  =>  '<i class="fas fa-info-circle"></i> How to find the WHMCS API Secret. Visit <a href="https://akdesigner.com/whmcs-templates/index.php?rp=/knowledgebase/53/Configure-WHMCS-API-Key-for-Hostiko.html#whmcs-api-credentials" target="_blank">API Secret</a>.',
						),
					),
				),
			);
		}

		CSF::createSection(
			'akd-framework-admin-page',
			array(
				'title'  => esc_html( '404 page' ),
				'icon'   => 'fas fa-file-alt',
				'class'	=> 'akd-title-options',
				'fields' => array(
					array(
						'type'    => 'heading',
						'content' => esc_html( '404 page' ),
						'icon'   => 'fas fa-file-alt',
						'class'   => 'akd-support-section-title akd-title-options',
					),
					array(
						'id'          => 'opt-select-404-page',
						'type'        => 'select',
						'title'       => 'Select 404 Page',
						'placeholder' => 'Select a page',
						'options'     => $hostiko_elementor_template_list,
						'default'     => 'option-2'
					),
					array(
						'id'      => 'opt-404-title',
						'type'    => 'text',
						'title'   => '400 Text',
						'default' => '404',
						'dependency' => array( 'opt-select-404-page', '==', '' ),
					),
					array(
						'id'      => 'opt-404-not-found-title',
						'type'    => 'text',
						'title'   => 'Not Found Title',
						'default' => 'Page not found',
						'dependency' => array( 'opt-select-404-page', '==', '' ),
					),
					array(
						'id'      => 'opt-404-desc',
						'type'    => 'textarea',
						'title'   => '404 Description Text',
						'default' => 'Oops!! The page you are looking for does not exist. It might have been moved or deleted',
						'dependency' => array( 'opt-select-404-page', '==', '' ),
					  ),
					array(
						'id'      => 'opt-404-link-text',
						'type'    => 'text',
						'title'   => '404 Link Text',
						'default' => 'Back To Home',
						'dependency' => array( 'opt-select-404-page', '==', '' ),
					),
				),
			),
		);

		CSF::createSection(
			'akd-framework-admin-page',
			array(
				'title'  => esc_html( 'Help & Support' ),
				'icon'   => 'fas fa-info-circle',
				'fields' => array(
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Live Customer Support' ),
						'class'   => 'akd-support-section-title akd-title-options',
					),
					array(
						'type'    => 'content',
						'content' => '<h2 class="customer-support-title">Welcome to our Live Customer Support! </h2><p>You can now chat with our live customer care representatives for immediate assistance. To start a conversation, click the link below:</p> <a target=_blank href="https://designingmedia.com/help-center" class="akd-live-support akd-cdf-button">' . esc_html( 'Start Live Chat' ) . '</a>'
					),
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Note:' ),
						'class'   => 'akd-support-section-title akd-sub-heading',
					),
					array(
						'type'    => 'content',
						'class'   => 'live-support-notice akd-notice-box',
						'content' => '<p>' . esc_html( '- Live support is available only to users with valid support subscriptions.' ) . '</p>
							<p>' . esc_html( '- Our support hours are from 10:00 AM to 7:00 PM (GMT) UTC +5, Monday to Friday.' ) . '</p>
							<p>' . 'If you are contacting us outside of support hours or if all our agents are currently busy, we recommend opening a ticket at our support portal: <a target=_blank href="https://designingmedia.com/support">https://designingmedia.com/support</a>' . '</p>'
					),
					array(
						'type'    => 'heading',
						'content' => esc_html( 'Frequently Asked Question' ),
						'class'   => 'akd-support-section-title akd-sub-heading',
					),
					array(
						'id'         => 'opt-accordion-2',
						'type'       => 'accordion',
						'class'      => 'akd-support-accordion-options',
						'accordions' => array(
							array(
								'title'  => esc_html( 'Q: Customization and Theme Setup Services' ),
								'fields' => array(
									array(
										'id'      => 'opt-text-1',
										'type'    => 'content',
										'content' => 'If you are looking to customize the theme according to your needs or want us to install the theme and set up everything for you, we offer additional services to help with that. Please reach out to our support team through the <a target=_blank href="https://designingmedia.com/support"> https://designingmedia.com/support</a> to discuss your requirements and get a quote for the services.',
									),
								),
							),
							array(
								'title'  => esc_html( 'Q: What is needed before submitting a support ticket?' ),
								'fields' => array(
									array(
										'id'      => 'opt-text-2',
										'type'    => 'content',
										'content' => '<p>' . esc_html( '1. Ensure you are running the latest versions of WordPress.' ) . '</p>
													<p>' . esc_html( '2. <b>Check our Knowledgebase</b> for the answer to your question.' ) . '</p>
													<p>' . esc_html( '3. <b>Disable any additional 3rd party plugins</b> and see whether it fixes the issue.' ) . '</p>
													<p>' . esc_html( '4. Please make sure you <b>backup your entire site first</b> – either via cPanel or manually back up your WordPress, uploads folders, as well as your WordPress database. Your web hosting support can guide you more about backup options on your hosting account.' ) . '</p>
													<p>' . esc_html( '5. We recommend <b>creating a staging site</b> for our dev team or for yourself while developing so in the worst situation it doesn’t affect your live site. A staging site is a clone of your live website that enables you to test any changes or major new features in a secure environment. Developers typically use staging sites to test changes and fix bugs before going to production.' ) . '</p>',
									),
								),
							),
							array(
								'title'  => esc_html( 'Q: What is the scope of support?' ),
								'fields' => array(
									array(
										'id'      => 'opt-text-2',
										'type'    => 'content',
										'content' => '<p>' . esc_html( 'Our support covers:' ) . '</p>
												<p>' . esc_html( '- Getting setup.' ) . '</p>
												<p>' . esc_html( '- Troubleshooting any features.' ) . '</p>
												<p>' . esc_html( '- Fixing any bugs that may arise.t</b> – either via cPanel or manually back up your WordPress, uploads folders, as well as your WordPress database. Your web hosting support can guide you more about backup options on your hosting account.' ) . '</p>
												<p>' . esc_html( 'Unfortunately, we cannot provide support for 3rd party plugins or customizations. You need to enlist the help of a developer for customizations of your theme.' ) . '</p>',
									),
								),
							),
							array(
								'title'  => esc_html( 'Q: What is the support schedule?' ),
								'fields' => array(
									array(
										'id'      => 'opt-text-2',
										'type'    => 'content',
										'content' => '<p>' . esc_html( 'We do our best to monitor the tickets around the clock, however, this is not always possible due to different time zones.' ) . '</p>
													<p>' . esc_html( 'Reply times can vary from time to time and be up to 24 hours Monday – Friday and up to 48 hours on weekends (Sat-Sun), but are usually faster than that.' ) . '</p>
													<p>' . esc_html( 'Please be sure to wait for our reply. We answer threads from oldest to newest, so if you bump the thread before we get to it, it will throw it back to the beginning of the queue and take longer. We will answer it if we have not already.' ) . '</p>
													<p>' . esc_html( '(We are open from 10am to 7pm (GMT) UTC +5, Monday to Friday.)' ) . '</p>',
									),
								),
							),
						),
					),
				),
			),
		);
		
	}

	/**
	 * Method to load all required plugins by the theme
	 */
	public function akd_required_plugins_callback() {
		if ( class_exists( 'TGM_Plugin_Activation' ) ) {
			/**
			 * Check if TGMPA is initialized
			 */
			if ( isset( $GLOBALS['tgmpa'] ) ) {
				/**
				 * Get TGMPA instance
				 */
				$tgmpa_instance = $GLOBALS['tgmpa'];
				if ( TGM_Plugin_Activation::get_instance()->is_tgmpa_complete() == false ) {
					$content = '<div class="akd-required-plugins-section">
							<p class="akd-required-plugin-info-box-notice">' . esc_html( 'If you encounter any errors or if the plugins are not visible to you, please open a support ticket at <a target=_blank; href="https://designingmedia.com/support">https://designingmedia.com/support</a>.' ) . '</p>
						</div>
					';
					echo $content;
				}
			}
		}

		$merlin = new Merlin;
		$merlin->plugins();
	}

	/**
	 * Method to load theme's demo data
	 */
	public function akd_theme_demo_importer_callback() {
		$content       = '';
		$demo_dir      = plugin_dir_path( dirname( __FILE__ ) ) . 'admin/demo/';
		$screenshot    = $demo_dir . 'screenshot.png';
		$layout_number = '';

		if ( ! get_option( '_akd_imported_layout' ) ) {
			$content .= '<div class="akd-required-plugins-section">
					<p class="akd-required-plugin-info-box-notice">' . esc_html( 'If you encounter any errors or if the demos are not visible to you, please open a support ticket: <a target= _blank href="https://designingmedia.com/support"> https://designingmedia.com/support/</a>.' ) . '</p>
				</div>
			';
		}

		if ( is_dir( $demo_dir ) ) {
			$files = array_diff( scandir( $demo_dir ), array( '.', '..' ) );
			/**
			 * Hostiko templates
			 */
			if ( count( $files ) === 1 && in_array( 'index.php', $files ) ) {
				$base_url = get_option( '_akd_download_url' ) . 'Hostiko/';
				$json_url = $base_url . 'akd-importer-data-listing.json';

				// $json_content = file_get_contents( $json_url );
				$curl = curl_init();

				curl_setopt_array( $curl, array(
					CURLOPT_URL            => $json_url,
					CURLOPT_RETURNTRANSFER => true,
					CURLOPT_ENCODING       => '',
					CURLOPT_MAXREDIRS      => 10,
					CURLOPT_TIMEOUT        => 5,
					CURLOPT_FOLLOWLOCATION => true,
					CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
					CURLOPT_CUSTOMREQUEST  => 'GET',
				));

				$json_content = curl_exec( $curl );

				curl_close( $curl );

				/**
				 * Check if the content was fetched successfully
				 */
				if ( FALSE === $json_content ) {
					$content .= esc_html( 'Failed to retrieve the content.' );
				} else {
					/**
					 * Optionally, decode the JSON content into an associative array
					 */
					$data = json_decode( $json_content, true );
					$data = array_reverse( $data );

					/**
					 * Check if the JSON decoding was successful
					 */
					if ( $data === NULL ) {
						$content .= '<span class="akd-demos">' . esc_html( 'No demos available.' ) . '</span>';
					} else {

						/**
						 * Manual Import
						 */

						// $content .= '
						// 	<div class="akd-required-plugins-section">
						// 		<p class="akd-required-plugin-info-box-notice">'.esc_html('Select your desired demo, and then click "Install" to initiate the installation of the demo data.').'</p>	
						// 	</div>

						// 	<div class="akd-manual-import">
						// 		<h5 class="akd-manual-import-title">
						// 			'.esc_html('Installing Demo Content Manually').'
						// 		</h5>
						// 		<p>'.esc_html('If you encounter any issues importing demo data, you always have the option to upload it manually:').'</p>

						// 		<label for="layout_select">Select your layout to download: </label>
						// 		<select name="layout_select" id="layout_select">';

						// 			foreach($data as $akd_demo_key => $demo_data){
						// 				$content .= '<option value="' . $demo_data['scr'] . '">' . $demo_data['title'] . '</option>';
						// 			}

						// 		$content .= '</select>';

						// 		$content .= '<ol>
						// 			<li> '.esc_html('Click to download ').' <a href="layout-65" class="akd-download-manual-files" target="_blank">'.esc_html('this zip file.').'</a></li>
						// 			<li> '.esc_html('Upload the downloaded zip file to "demo" folder within the akd-demo-importer plugin, e.g. /wp-content/plugins/akd-demo-importer/admin/demo').' </li>
						// 			<li> '.esc_html('Revisit this page, refresh it, and then proceed with another attempt to import the demo.').' </li>
						// 		</ol>
						// 	</div>
						// ';

						/**
						 * Output the JSON data (or handle it as needed)
						 */
						$content .= '<div class="demo-container">';
							foreach( $data as $akd_demo_key => $demo_data ) {
								$class = '';
								if ( get_option( '_akd_imported_layout' ) && get_option( '_akd_imported_layout' ) == explode( ' ', $demo_data['title'] )[1] ) {
									$class = 'import-layout';
								} elseif ( get_option( '_akd_imported_layout' ) && get_option( '_akd_imported_layout' ) != explode( ' ', $demo_data['title'] )[1] ) {
									$class = 'not-imported-layout';
								} else {
									$class = '';
								}
								$content .= '
									<div class="demo-item ' . $class . '">
										<div class="screenshot-container">
											<img class="screenshot" src="' . get_option( '_akd_download_url' ) .'Hostiko/' . $demo_data['scr'] . '.png' . '" alt="' . $demo_data['title'] . '">
											<div class="overlay"></div>';
											if ( ! get_option( '_akd_imported_layout' ) ) {
												$content .= '
													<div class="button-container">
														<a class="preview-button" href="https://hostiko.com/layout' . explode(' ', $demo_data['title'])[1] . '"target="_blank">'.esc_html('View Demo').'</a>
														<a class="proceed-button akd-trigger-import" data-layout="'.explode(' ', $demo_data['title'])[1].'" href="#" >'.esc_html('Install Demo').'</a>
													</div>
												';
											} else if ( explode( ' ', $demo_data['title'])[1] == get_option( '_akd_imported_layout' ) ) {
												$content .= '
													<div class="button-container">
														<a class="preview-button" href="/" target="_blank">'.esc_html('View Site').'</a>
														<a class="proceed-button akd-trigger-import" data-layout="'.explode(' ', $demo_data['title'])[1].'" href="#" >'.esc_html('Update Demo').'</a>
													</div>
												';
											} else {
												$content .= '
													<div class="button-container">
														<a class="preview-button" href="https://hostiko.com/layout' . explode(' ', $demo_data['title'])[1] . '" target="_blank">'.esc_html('View Demo').'</a>
														<a class="proceed-button akd-new-damo-triger" data-layout="'.explode(' ', $demo_data['title'])[1].'" href="#" >'.esc_html('Install Demo').'</a>
													</div>
												';
											}
										$content .= '</div>
										<div class="layout-name">' . $demo_data['title'] . '</div>
									</div>
								';
							}
						$content .= '</div>';

						if ( get_option( '_akd_demo_import' ) ) {
							$content .= '<a href="" class="akd-content-next"> ' . esc_html( 'Next' ) . ' </a>';
						}
					}
				}
			} else {
				/**
				 * Single layout
				 */
				if( ! get_option( '_akd_demo_import' ) ) {
					if ( ! empty( $screenshot ) ) {
						$content .= '<div class="demo-container">';
							$demo_name       = basename( dirname( $screenshot ) );
							$screenshot      = plugins_url( 'akd-demo-importer/admin/demo/screenshot.png' );
							$activated_theme = wp_get_theme();

							$content .= '
								<div class="demo-item">
									<div class="screenshot-container">
										<img class="screenshot" src="' . $screenshot . '" alt="' . $activated_theme->get_template() . '">
										<div class="overlay"></div>
										<div class="button-container">
											<a class="preview-button" href="https://designingmedia.com/' . $activated_theme->get_template() . '" target="_blank">' . esc_html( 'Preview' ) . '</a>
											<a class="proceed-button akd-trigger-single-import" href="#" >' . esc_html( 'Install' ) . '</a>
										</div>
									</div>
									<div class="layout-name">' . ucfirst( $activated_theme->get_template() ) . '</div>
								</div>
							';
						$content .= '</div>';
					} else {
						$content .= '<span class="akd-demos">' . esc_html( 'No demos available.' ) . '</span>';
					}
				} else {
					$content .= '
						<div class="merlin__content merlin__content--plugins">
							<div class="merlin__content--transition akd_content--transition">
								<img src=' . plugins_url( 'akd-demo-importer/includes/merlin/assets/images/import.svg' ) . ' alt="plugin_img">
								<h1>' . esc_html( 'You are up to speed!' ) . '</h1>
								<p>' . esc_html( 'The demo has been imported successfully.' ) . '</p>
		
								<div class="akd-view-site-wrapper">
									<a class="merlin__button akd-view-site" href="#"> ' . esc_html( 'View Site' ) . ' </a>
								</div>
							</div>
						</div>
					';
				}
			}
		}

		echo $content;
	}

	/**
	 * Method to show merlin import layout
	 */
	public function akd_install_demo_data_callback() {

		if ( ! get_option( '_akd_imported_layout' ) ) {
			$content = '<div class="akd-required-plugins-section">
					<p class="akd-required-plugin-info-box-notice">' . esc_html( 'If you encounter any errors or if the plugins are not visible to you, please open a support ticket at <a target=_blank; href="https://designingmedia.com/support">https://designingmedia.com/support</a>.' ) . '</p>
				</div>
			';

			echo $content;
		}

		$merlin = new Merlin;
		$merlin->content();

		$content = '';
		$content .= '<div class="akd-demo-import-content">

		</div>';
	}

	
	private function akd_get_theme_logo(){
		$theme = wp_get_theme();
		$theme_slug = $theme->get_stylesheet();
		$akd_theme_mods = get_option('theme_mods_'.$theme_slug);

		// Check if the theme modifications array exists and if the logo is set
		if (!empty($akd_theme_mods) && isset($akd_theme_mods['logo'])) {
			// Get the logo ID
			$akd_logo_url = $akd_theme_mods['logo'];
			return $akd_logo_url;
		} 
		return '';
	}
	private function akd_get_theme_favicon(){
		// Get the Elementor global settings
		$site_icon_url = get_site_icon_url();
		if ($site_icon_url) {
			return $site_icon_url;
		}
		return '';
	}


	
 	/**
	 * Method to get and present the elementor color settings within theme settings tabs
	  */
	private function akd_get_elementor_theme_color_settings() {
		
		if ( in_array( 'elementor/elementor.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {

			/**
			 * Path to the JSON file
			 */
			$font_file_path = plugin_dir_path( __FILE__ ) . 'class-akd-framework-google-font-list.json';
			/**
			 * Read the JSON file content
			 */
			$json_content = file_get_contents( $font_file_path );
			/**
			 * Decode the JSON string into a PHP array
			 */
			$fonts_array = json_decode( $json_content, true );

			$active_kit_id             = get_option( 'elementor_active_kit' );
			$elementor_global_settings = get_option( '_akd_elementor_global' );
			

			if ( $active_kit_id ) {
				$elementor_data = get_post_meta( $active_kit_id, '_elementor_page_settings', true );
				

				if ( $elementor_data ) {
					$custom_colors_options = array();
					$custom_colors_default = array();
					$options               = array();
					$default_colors        = array();

					foreach ( $elementor_data['system_colors'] as $color_data ) {
						$color_id    = $color_data['_id'];
						$color_title = ucfirst( $color_data['title'] );
						$color_key   = str_replace( '-', '_', $color_id );

						/**
						 * Add color options for regular and hovered states
						 */
						$options[$color_key] = $color_title;

						/**
						 * Add default colors
						 */
						$default_colors[$color_key] = $color_data['color'];
					}

					if ( isset( $elementor_data['custom_colors'] ) ) {
						foreach ( $elementor_data['custom_colors'] as $color_data ) {
							$color_id    = $color_data['_id'] ?? '';
							$color_title = ucfirst( $color_data['title'] ?? 'Untitled' );
							$color_key   = 'custom-color-' . $color_id;
					
							// Only proceed if 'color' key exists
							if ( isset( $color_data['color'] ) ) {
								$color = $color_data['color'];
					
								// Add color options for regular and hovered states
								$custom_colors_options[$color_key] = $color_title;
					
								// Add default colors
								$custom_colors_default[$color_key] = $color;
							}
						}
					}					

					$theme_settings = array(
						'title'  => esc_html( 'Theme Color' ),
						'icon'   => 'fas fa-palette',
						'fields' => array(
							array(
								'type'    => 'subheading',
								'content' => esc_html( 'Global Colors' ),
								'class'   => 'akd-support-section-title akd-title-options'
							),
							array(
								'id'      => 'akd-theme-colors',
								'type'    => 'color_group',
								'title'   => esc_html( 'Theme Colors' ),
								'options' => $options,
								'default' => $default_colors,
							),
							/**
							 * Custom colors
							 */
							array(
								'id'      => 'akd-custom-colors',
								'type'    => 'color_group',
								'title'   => esc_html( 'Custom Colors' ),
								'options' => $custom_colors_options,
								'default' => $custom_colors_default,
							),
						),
					);
				} else {
					$theme_settings = array(
						'title'  => esc_html( 'Theme Settings' ),
						'icon'   => 'fas fa-cog',
						'fields' => array(
							array(
								'type'    => 'notice',
								'style'   => 'error',
								'content' => esc_html( 'Please import demo content to make the theme settings enable.' ),
							),
						),
					);
				}
			} else {
				$theme_settings = array(
					'title'  => esc_html( 'Theme Settings' ),
					'icon'   => 'fas fa-cog',
					'fields' => array(
						array(
							'type'    => 'notice',
							'style'   => 'error',
							'content' => esc_html( 'Please import demo content to make the theme settings enable.' ),
						),
					),
				);
			}
		} else {
			$theme_settings = array(
				'title'  => esc_html( 'Theme Settings' ),
				'icon'   => 'fas fa-cog',
				'fields' => array(
					
					array(
						'type'    => 'subheading',
						'content' => esc_html( 'Global Colors' ),
					),
					array(
						'type'    => 'subheading',
						'content' => esc_html( 'Typography' ),
					),
				),
			);
		}

		return $theme_settings;
	}

	private function akd_get_elementor_theme_typography_settings() {
		if (in_array('elementor/elementor.php', apply_filters('active_plugins', get_option('active_plugins')))) {
	
			// Load Google Fonts JSON file
			$font_file_path = plugin_dir_path(__FILE__) . 'class-akd-framework-google-font-list.json';
			$json_content = file_get_contents($font_file_path);
			$fonts_array = json_decode($json_content, true);
	
			$active_kit_id = get_option('elementor_active_kit');
	
			if ($active_kit_id) {
				$elementor_data = get_post_meta($active_kit_id, '_elementor_page_settings', true);
	
				if ($elementor_data) {
					$theme_settings = array(
						'title'  => esc_html('Theme Typography'),
						'icon'   => 'fas fa-pen-nib',
						'fields' => array(),
					);
	
					// Text transform options
					$text_transform_options = array(
						'none' => esc_html('None'),
						'capitalize' => esc_html('Capitalize'),
						'uppercase' => esc_html('Uppercase'),
						'lowercase' => esc_html('Lowercase'),
						'initial' => esc_html('Initial'),
						'inherit' => esc_html('Inherit'),
					);
	
					// Grouped typography fields
					foreach ($elementor_data['system_typography'] as $typography) {
						$typography_id = 'akd-' . $typography['_id'];
	
						// Group for each typography
						$theme_settings['fields'][] = array(
							'type'    => 'subheading',
							'content' => esc_html($typography['title']),
							'class'   => 'akd-support-section-title akd-title-options'
						);
	
						// Primary Font
						$theme_settings['fields'][] = array(
							'id'          => $typography_id . '-typography',
							'type'        => 'select',
							'title'       => esc_html('Primary Font'),
							'chosen'      => true,
							'ajax'        => true,
							'placeholder' => esc_html('Select an option'),
							'options'     => $fonts_array,
							'default'     => $typography['typography_font_family'] ?? 'default',
						);
	
						// Font Weight
						$theme_settings['fields'][] = array(
							'id'      => $typography_id . '-font-weight',
							'type'    => 'number',
							'title'   => esc_html('Font Weight'),
							'default' => $typography['typography_font_weight'] ?? '',
						);
	
						// Font Size
						$theme_settings['fields'][] = array(
							'id'      => $typography_id . '-font-size',
							'type'    => 'number',
							'title'   => esc_html('Font Size'),
							'default' => $typography['typography_font_size']['size'] ?? '',
						);
	
						// Text Transform
						$theme_settings['fields'][] = array(
							'id'      => $typography_id . '-text-transform',
							'type'    => 'select',
							'title'   => esc_html('Text Transform'),
							'options' => $text_transform_options,
							'default' => $typography['typography_text_transform'] ?? 'none',
						);
					}
	
					// Append custom typography fields if available
					if (isset($elementor_data['custom_typography'])) {
						$theme_settings['fields'][] = array(
							'type'    => 'subheading',
							'content' => esc_html('Custom Typography'),
							
						);
						foreach ($elementor_data['custom_typography'] as $custom_typography) {
							$custom_typography_id = 'akd-' . $custom_typography['_id'];

							// Group for each typography
							$theme_settings['fields'][] = array(
								'type'    => 'subheading',
								'content' => esc_html($custom_typography['title']),
							);
							// Custom Font
							$theme_settings['fields'][] = array(
								'id'          => $custom_typography_id . '-typography',
								'type'        => 'select',
								'title'       => esc_html($custom_typography['title']),
								'chosen'      => true,
								'ajax'        => true,
								'placeholder' => esc_html('Select an option'),
								'options'     => $fonts_array,
								'default'     => $custom_typography['typography_font_family'],
							);
							// Custom Font Weight
							$theme_settings['fields'][] = array(
								'id'      => $custom_typography_id . '-font-weight',
								'type'    => 'number',
								'title'   => esc_html('Font Weight'),
								'default' => $custom_typography['typography_font_weight'] ?? '',
							);
							// Custom Font Size
							$theme_settings['fields'][] = array(
								'id'      => $custom_typography_id . '-font-size',
								'type'    => 'number',
								'title'   => esc_html('Font Size'),
								'default' => $custom_typography['typography_font_size']['size'] ?? '',
							);
	
							// Custom Text Transform
							$theme_settings['fields'][] = array(
								'id'      => $custom_typography_id . '-text-transform',
								'type'    => 'select',
								'title'   => esc_html('Text Transform'),
								'options' => $text_transform_options,
								'default' => $custom_typography['typography_text_transform'] ?? 'none',
							);
						}
					}
				} else {
					$theme_settings = $this->get_demo_import_notice();
				}
			} else {
				$theme_settings = $this->get_demo_import_notice();
			}
		} else {
			$theme_settings = array(
				'title'  => esc_html('Theme Typography'),
				'icon'   => 'fas fa-pen-nib',
				'fields' => array(
					array(
						'type'    => 'subheading',
						'content' => esc_html('Typography'),
					),
				),
			);
		}
	
		return $theme_settings;
	}
	
	/**
	 * Helper function to create the demo import notice
	 */
	private function get_demo_import_notice() {
		return array(
			'title'  => esc_html( 'Theme Typography' ),
			'icon'   => 'fas fa-pen-nib',
			'fields' => array(
				array(
					'type'    => 'notice',
					'style'   => 'error',
					'content' => esc_html( 'Please import demo content to make the theme settings enable.' ),
				),
			),
		);
	}
	
	/**
	 * Method to create metabox
	 *
	 * @return void
	 */
	private function init_meta() {
		CSF::createMetabox(
			$this->prefix,
			array(
				'title'     => esc_html( $this->meta_title ),
				'post_type' => $this->post_type,
			)
		);
	}

	/**
	 * Method to populate the metabox
	 *
	 * @return void
	 */
	private function introduce_tabs() {
		$tabs = $this->get_fields();
		foreach ( $tabs as $tab ) {
			CSF::createSection( $this->prefix, $tab );
		}
	}
}